/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.dataprepper.plugins.processor.oteltrace;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import io.micrometer.core.instrument.util.StringUtils;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;
import java.util.stream.Collectors;
import org.opensearch.dataprepper.metrics.PluginMetrics;
import org.opensearch.dataprepper.model.annotations.DataPrepperPlugin;
import org.opensearch.dataprepper.model.annotations.DataPrepperPluginConstructor;
import org.opensearch.dataprepper.model.configuration.PipelineDescription;
import org.opensearch.dataprepper.model.peerforwarder.RequiresPeerForwarding;
import org.opensearch.dataprepper.model.processor.AbstractProcessor;
import org.opensearch.dataprepper.model.processor.Processor;
import org.opensearch.dataprepper.model.record.Record;
import org.opensearch.dataprepper.model.trace.Span;
import org.opensearch.dataprepper.plugins.processor.oteltrace.OtelTraceRawProcessorConfig;
import org.opensearch.dataprepper.plugins.processor.oteltrace.model.SpanSet;
import org.opensearch.dataprepper.plugins.processor.oteltrace.model.TraceGroup;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@DataPrepperPlugin(name="otel_traces", deprecatedName="otel_trace_raw", pluginType=Processor.class, pluginConfigurationType=OtelTraceRawProcessorConfig.class)
public class OTelTraceRawProcessor
extends AbstractProcessor<Record<Span>, Record<Span>>
implements RequiresPeerForwarding {
    private static final long SEC_TO_MILLIS = 1000L;
    private static final Logger LOG = LoggerFactory.getLogger(OTelTraceRawProcessor.class);
    public static final String TRACE_GROUP_CACHE_COUNT_METRIC_NAME = "traceGroupCacheCount";
    public static final String SPAN_SET_COUNT_METRIC_NAME = "spanSetCount";
    private final long traceFlushInterval;
    private final Map<String, SpanSet> traceIdSpanSetMap = new ConcurrentHashMap<String, SpanSet>();
    private final Cache<String, TraceGroup> traceIdTraceGroupCache;
    private long lastTraceFlushTime = 0L;
    private final ReentrantLock traceFlushLock = new ReentrantLock();
    private final ReentrantLock prepareForShutdownLock = new ReentrantLock();
    private volatile boolean isShuttingDown = false;

    @DataPrepperPluginConstructor
    public OTelTraceRawProcessor(OtelTraceRawProcessorConfig otelTraceRawProcessorConfig, PipelineDescription pipelineDescription, PluginMetrics pluginMetrics) {
        super(pluginMetrics);
        this.traceFlushInterval = 1000L * otelTraceRawProcessorConfig.getTraceFlushIntervalSeconds();
        this.traceIdTraceGroupCache = Caffeine.newBuilder().maximumSize(otelTraceRawProcessorConfig.getTraceGroupCacheMaxSize()).expireAfterWrite(otelTraceRawProcessorConfig.getTraceGroupCacheTimeToLive().toMillis(), TimeUnit.MILLISECONDS).build();
        pluginMetrics.gauge(TRACE_GROUP_CACHE_COUNT_METRIC_NAME, this.traceIdTraceGroupCache, cache -> cache.estimatedSize());
        pluginMetrics.gauge(SPAN_SET_COUNT_METRIC_NAME, this.traceIdSpanSetMap, cache -> cache.size());
        LOG.info("Configured Trace Raw Processor with a trace flush interval of {} ms.", (Object)this.traceFlushInterval);
    }

    public Collection<Record<Span>> doExecute(Collection<Record<Span>> records) {
        LinkedList<Span> processedSpans = new LinkedList<Span>();
        for (Record<Span> record : records) {
            Span span = (Span)record.getData();
            this.processSpan(span, processedSpans);
            this.fillInServiceName(span);
        }
        processedSpans.addAll(this.getTracesToFlushByGarbageCollection());
        return processedSpans.stream().map(Record::new).collect(Collectors.toList());
    }

    private void processSpan(Span span, Collection<Span> spanSet) {
        if (StringUtils.isBlank((String)span.getParentSpanId())) {
            List<Span> rootSpanAndChildren = this.processRootSpan(span);
            spanSet.addAll(rootSpanAndChildren);
        } else {
            Optional<Span> populatedChildSpanOptional = this.processChildSpan(span);
            populatedChildSpanOptional.ifPresent(spanSet::add);
        }
    }

    private List<Span> processRootSpan(Span parentSpan) {
        TraceGroup traceGroup = TraceGroup.fromSpan(parentSpan);
        String parentSpanTraceId = parentSpan.getTraceId();
        this.traceIdTraceGroupCache.put((Object)parentSpanTraceId, (Object)traceGroup);
        LinkedList<Span> recordsToFlush = new LinkedList<Span>();
        recordsToFlush.add(parentSpan);
        SpanSet spanSet = this.traceIdSpanSetMap.get(parentSpanTraceId);
        if (spanSet != null) {
            for (Span span : spanSet.getSpans()) {
                this.fillInTraceGroupInfo(span, traceGroup);
                recordsToFlush.add(span);
            }
            this.traceIdSpanSetMap.remove(parentSpanTraceId);
        }
        return recordsToFlush;
    }

    private Optional<Span> processChildSpan(Span childSpan) {
        String childSpanTraceId = childSpan.getTraceId();
        TraceGroup traceGroup = (TraceGroup)this.traceIdTraceGroupCache.getIfPresent((Object)childSpanTraceId);
        if (traceGroup != null) {
            this.fillInTraceGroupInfo(childSpan, traceGroup);
            return Optional.of(childSpan);
        }
        this.traceIdSpanSetMap.compute(childSpanTraceId, (traceId, spanSet) -> {
            if (spanSet == null) {
                spanSet = new SpanSet();
            }
            spanSet.addSpan(childSpan);
            return spanSet;
        });
        return Optional.empty();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private List<Span> getTracesToFlushByGarbageCollection() {
        boolean isLockAcquired;
        LinkedList<Span> recordsToFlush = new LinkedList<Span>();
        if (this.shouldGarbageCollect() && (isLockAcquired = this.traceFlushLock.tryLock())) {
            try {
                long now;
                this.lastTraceFlushTime = now = System.currentTimeMillis();
                Iterator<Map.Entry<String, SpanSet>> entryIterator = this.traceIdSpanSetMap.entrySet().iterator();
                while (entryIterator.hasNext()) {
                    Map.Entry<String, SpanSet> entry = entryIterator.next();
                    String traceId = entry.getKey();
                    TraceGroup traceGroup = (TraceGroup)this.traceIdTraceGroupCache.getIfPresent((Object)traceId);
                    SpanSet spanSet = entry.getValue();
                    long traceTime = spanSet.getTimeSeen();
                    if (now - traceTime < this.traceFlushInterval && !this.isShuttingDown) continue;
                    Set<Span> spans = spanSet.getSpans();
                    if (traceGroup != null) {
                        spans.forEach(span -> {
                            this.fillInTraceGroupInfo((Span)span, traceGroup);
                            this.fillInServiceName((Span)span);
                            recordsToFlush.add((Span)span);
                        });
                    } else {
                        LOG.warn("There are {} spans with missing trace groups. Unable to populate with trace group information.", (Object)spans.size());
                        spans.forEach(span -> {
                            recordsToFlush.add((Span)span);
                            LOG.debug("Missing trace group for SpanId: {}", (Object)span.getSpanId());
                        });
                    }
                    entryIterator.remove();
                }
                if (!recordsToFlush.isEmpty()) {
                    LOG.info("Flushing {} records", (Object)recordsToFlush.size());
                }
            }
            finally {
                this.traceFlushLock.unlock();
            }
        }
        return recordsToFlush;
    }

    private void fillInTraceGroupInfo(Span span, TraceGroup traceGroup) {
        span.setTraceGroup(traceGroup.getTraceGroup());
        span.setTraceGroupFields(traceGroup.getTraceGroupFields());
    }

    private void fillInServiceName(Span span) {
        span.setServiceName(span.getServiceName());
    }

    private boolean shouldGarbageCollect() {
        return System.currentTimeMillis() - this.lastTraceFlushTime >= this.traceFlushInterval || this.isShuttingDown;
    }

    public void prepareForShutdown() {
        boolean isLockAcquired = this.prepareForShutdownLock.tryLock();
        if (isLockAcquired) {
            try {
                LOG.info("Preparing for shutdown, will attempt to flush {} spans", (Object)this.traceIdSpanSetMap.size());
                this.isShuttingDown = true;
            }
            finally {
                this.prepareForShutdownLock.unlock();
            }
        }
    }

    public boolean isReadyForShutdown() {
        return this.traceIdSpanSetMap.isEmpty();
    }

    public void shutdown() {
        this.traceIdTraceGroupCache.cleanUp();
    }

    public Collection<String> getIdentificationKeys() {
        return Collections.singleton("traceId");
    }
}

