/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.dataprepper.plugins.sourcecoordinator.dynamodb;

import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import org.opensearch.dataprepper.model.source.coordinator.SourcePartitionStatus;
import org.opensearch.dataprepper.model.source.coordinator.SourcePartitionStoreItem;
import org.opensearch.dataprepper.model.source.coordinator.exceptions.PartitionUpdateException;
import org.opensearch.dataprepper.plugins.sourcecoordinator.dynamodb.DynamoDbSourcePartitionItem;
import org.opensearch.dataprepper.plugins.sourcecoordinator.dynamodb.DynamoStoreSettings;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.core.internal.waiters.ResponseOrException;
import software.amazon.awssdk.core.pagination.sync.SdkIterable;
import software.amazon.awssdk.enhanced.dynamodb.DynamoDbEnhancedClient;
import software.amazon.awssdk.enhanced.dynamodb.DynamoDbIndex;
import software.amazon.awssdk.enhanced.dynamodb.DynamoDbTable;
import software.amazon.awssdk.enhanced.dynamodb.Expression;
import software.amazon.awssdk.enhanced.dynamodb.Key;
import software.amazon.awssdk.enhanced.dynamodb.TableSchema;
import software.amazon.awssdk.enhanced.dynamodb.model.CreateTableEnhancedRequest;
import software.amazon.awssdk.enhanced.dynamodb.model.DeleteItemEnhancedRequest;
import software.amazon.awssdk.enhanced.dynamodb.model.EnhancedGlobalSecondaryIndex;
import software.amazon.awssdk.enhanced.dynamodb.model.GetItemEnhancedRequest;
import software.amazon.awssdk.enhanced.dynamodb.model.Page;
import software.amazon.awssdk.enhanced.dynamodb.model.PageIterable;
import software.amazon.awssdk.enhanced.dynamodb.model.PutItemEnhancedRequest;
import software.amazon.awssdk.enhanced.dynamodb.model.QueryConditional;
import software.amazon.awssdk.enhanced.dynamodb.model.QueryEnhancedRequest;
import software.amazon.awssdk.services.dynamodb.DynamoDbClient;
import software.amazon.awssdk.services.dynamodb.model.AttributeValue;
import software.amazon.awssdk.services.dynamodb.model.ConditionalCheckFailedException;
import software.amazon.awssdk.services.dynamodb.model.DescribeTableRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeTableResponse;
import software.amazon.awssdk.services.dynamodb.model.DescribeTimeToLiveRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeTimeToLiveResponse;
import software.amazon.awssdk.services.dynamodb.model.Projection;
import software.amazon.awssdk.services.dynamodb.model.ProjectionType;
import software.amazon.awssdk.services.dynamodb.model.ProvisionedThroughput;
import software.amazon.awssdk.services.dynamodb.model.ResourceInUseException;
import software.amazon.awssdk.services.dynamodb.model.TimeToLiveSpecification;
import software.amazon.awssdk.services.dynamodb.model.TimeToLiveStatus;
import software.amazon.awssdk.services.dynamodb.model.UpdateTimeToLiveRequest;
import software.amazon.awssdk.services.dynamodb.waiters.DynamoDbWaiter;

class DynamoDbClientWrapper {
    private static final Logger LOG = LoggerFactory.getLogger(DynamoDbClientWrapper.class);
    private static final int DEFAULT_QUERY_LIMIT = 1000;
    static final String SOURCE_STATUS_COMBINATION_KEY_GLOBAL_SECONDARY_INDEX = "source-status";
    static final String TTL_ATTRIBUTE_NAME = "expirationTime";
    static final String ITEM_DOES_NOT_EXIST_EXPRESSION = "attribute_not_exists(sourceIdentifier) or attribute_not_exists(sourcePartitionKey)";
    static final String ITEM_EXISTS_AND_HAS_LATEST_VERSION = "attribute_exists(sourceIdentifier) and attribute_exists(sourcePartitionKey) and version = :v";
    private final DynamoDbEnhancedClient dynamoDbEnhancedClient;
    private final DynamoDbClient dynamoDbClient;
    private DynamoDbTable<DynamoDbSourcePartitionItem> table;

    private DynamoDbClientWrapper(DynamoDbClient dynamoDbClient, DynamoDbEnhancedClient dynamoDbEnhancedClient) {
        this.dynamoDbClient = dynamoDbClient;
        this.dynamoDbEnhancedClient = dynamoDbEnhancedClient;
    }

    static DynamoDbClientWrapper create(DynamoDbClient dynamoDbClient) {
        DynamoDbEnhancedClient dynamoDbEnhancedClient = DynamoDbEnhancedClient.builder().dynamoDbClient(dynamoDbClient).build();
        return new DynamoDbClientWrapper(dynamoDbClient, dynamoDbEnhancedClient);
    }

    public void initializeTable(DynamoStoreSettings dynamoStoreSettings, ProvisionedThroughput provisionedThroughput) {
        this.table = this.dynamoDbEnhancedClient.table(dynamoStoreSettings.getTableName(), (TableSchema)TableSchema.fromBean(DynamoDbSourcePartitionItem.class));
        if (!dynamoStoreSettings.skipTableCreation().booleanValue()) {
            try {
                CreateTableEnhancedRequest createTableEnhancedRequest = CreateTableEnhancedRequest.builder().provisionedThroughput(provisionedThroughput).globalSecondaryIndices(new EnhancedGlobalSecondaryIndex[]{EnhancedGlobalSecondaryIndex.builder().indexName(SOURCE_STATUS_COMBINATION_KEY_GLOBAL_SECONDARY_INDEX).provisionedThroughput(provisionedThroughput).projection((Projection)Projection.builder().projectionType(ProjectionType.ALL).build()).build()}).build();
                this.table.createTable(createTableEnhancedRequest);
            }
            catch (ResourceInUseException e) {
                LOG.info("The table creation for {} was already triggered by another instance of data prepper", (Object)dynamoStoreSettings.getTableName());
            }
        }
        try (DynamoDbWaiter dynamoDbWaiter = DynamoDbWaiter.builder().client(this.dynamoDbClient).build();){
            DescribeTableRequest describeTableRequest = (DescribeTableRequest)DescribeTableRequest.builder().tableName(dynamoStoreSettings.getTableName()).build();
            ResponseOrException response = dynamoDbWaiter.waitUntilTableExists(describeTableRequest).matched();
            DescribeTableResponse describeTableResponse = (DescribeTableResponse)response.response().orElseThrow(() -> new RuntimeException(String.format("DynamoDb Table %s could not be found.", dynamoStoreSettings.getTableName())));
            LOG.debug("DynamoDB table {} was created successfully for source coordination", (Object)describeTableResponse.table().tableName());
            if (Objects.nonNull(dynamoStoreSettings.getTtl())) {
                if (!dynamoStoreSettings.skipTableCreation().booleanValue()) {
                    this.dynamoDbClient.updateTimeToLive((UpdateTimeToLiveRequest)UpdateTimeToLiveRequest.builder().tableName(this.table.tableName()).timeToLiveSpecification((TimeToLiveSpecification)TimeToLiveSpecification.builder().attributeName(TTL_ATTRIBUTE_NAME).enabled(Boolean.valueOf(Objects.nonNull(dynamoStoreSettings.getTtl()))).build()).build());
                } else {
                    DescribeTimeToLiveResponse describeTimeToLiveResponse = this.dynamoDbClient.describeTimeToLive((DescribeTimeToLiveRequest)DescribeTimeToLiveRequest.builder().tableName(this.table.tableName()).build());
                    if (Objects.isNull(describeTimeToLiveResponse.timeToLiveDescription()) || !TTL_ATTRIBUTE_NAME.equals(describeTimeToLiveResponse.timeToLiveDescription().attributeName()) || !TimeToLiveStatus.ENABLED.equals((Object)describeTimeToLiveResponse.timeToLiveDescription().timeToLiveStatus())) {
                        throw new RuntimeException(String.format("TTL is set for the DynamoDb source coordination store, but the necessary TTL is not enabled on the table. To use TTL with source coordination to clean up COMPLETED partitions, the table must have TTL enabled on the %s attribute.", TTL_ATTRIBUTE_NAME));
                    }
                }
            }
        }
    }

    public boolean tryCreatePartitionItem(DynamoDbSourcePartitionItem dynamoDbSourcePartitionItem) {
        try {
            this.table.putItem(PutItemEnhancedRequest.builder(DynamoDbSourcePartitionItem.class).item((Object)dynamoDbSourcePartitionItem).conditionExpression(Expression.builder().expression(ITEM_DOES_NOT_EXIST_EXPRESSION).build()).build());
            return true;
        }
        catch (ConditionalCheckFailedException e) {
            return false;
        }
        catch (Exception e) {
            LOG.error("An exception occurred while attempting to create a DynamoDb partition item {}", (Object)dynamoDbSourcePartitionItem.getSourcePartitionKey());
            throw new PartitionUpdateException("Exception when trying to create partition item " + dynamoDbSourcePartitionItem.getSourcePartitionKey(), (Throwable)e);
        }
    }

    private boolean tryAcquirePartitionItem(DynamoDbSourcePartitionItem dynamoDbSourcePartitionItem) {
        try {
            this.tryUpdateItem(dynamoDbSourcePartitionItem);
            return true;
        }
        catch (Exception e) {
            return false;
        }
    }

    public void tryUpdatePartitionItem(DynamoDbSourcePartitionItem dynamoDbSourcePartitionItem) {
        try {
            this.tryUpdateItem(dynamoDbSourcePartitionItem);
        }
        catch (PartitionUpdateException e) {
            LOG.warn(e.getMessage(), (Throwable)e);
            throw e;
        }
    }

    private void tryUpdateItem(DynamoDbSourcePartitionItem dynamoDbSourcePartitionItem) {
        try {
            dynamoDbSourcePartitionItem.setVersion(dynamoDbSourcePartitionItem.getVersion() + 1L);
            this.table.putItem(PutItemEnhancedRequest.builder(DynamoDbSourcePartitionItem.class).item((Object)dynamoDbSourcePartitionItem).conditionExpression(Expression.builder().expression(ITEM_EXISTS_AND_HAS_LATEST_VERSION).expressionValues(Map.of(":v", (AttributeValue)AttributeValue.builder().n(String.valueOf(dynamoDbSourcePartitionItem.getVersion() - 1L)).build())).build()).build());
        }
        catch (ConditionalCheckFailedException e) {
            String message = String.format("ConditionalCheckFailed while updating partition %s. This partition item was either deleted from the dynamo table, or another instance of Data Prepper has modified it. Expected version: %s", dynamoDbSourcePartitionItem.getSourcePartitionKey(), dynamoDbSourcePartitionItem.getVersion() - 1L);
            throw new PartitionUpdateException(message, (Throwable)e);
        }
        catch (Exception e) {
            String errorMessage = String.format("An exception occurred while attempting to update a DynamoDb partition item %s", dynamoDbSourcePartitionItem.getSourcePartitionKey());
            LOG.error(errorMessage, (Throwable)e);
            throw new PartitionUpdateException(errorMessage, (Throwable)e);
        }
    }

    public void tryDeletePartitionItem(DynamoDbSourcePartitionItem dynamoDbSourcePartitionItem) {
        dynamoDbSourcePartitionItem.setVersion(dynamoDbSourcePartitionItem.getVersion() + 1L);
        try {
            this.table.deleteItem(DeleteItemEnhancedRequest.builder().key(Key.builder().partitionValue(dynamoDbSourcePartitionItem.getSourceIdentifier()).sortValue(dynamoDbSourcePartitionItem.getSourcePartitionKey()).build()).conditionExpression(Expression.builder().expression(ITEM_EXISTS_AND_HAS_LATEST_VERSION).expressionValues(Map.of(":v", (AttributeValue)AttributeValue.builder().n(String.valueOf(dynamoDbSourcePartitionItem.getVersion() - 1L)).build())).build()).build());
        }
        catch (Exception e) {
            String errorMessage = String.format("An exception occurred while attempting to delete a DynamoDb partition item %s", dynamoDbSourcePartitionItem.getSourcePartitionKey());
            LOG.error(errorMessage, (Throwable)e);
            throw new PartitionUpdateException(errorMessage, (Throwable)e);
        }
    }

    public Optional<SourcePartitionStoreItem> getSourcePartitionItem(String sourceIdentifier, String sourcePartitionKey) {
        try {
            Key key = Key.builder().partitionValue(sourceIdentifier).sortValue(sourcePartitionKey).build();
            SourcePartitionStoreItem result = (SourcePartitionStoreItem)this.table.getItem(GetItemEnhancedRequest.builder().key(key).build());
            if (Objects.isNull(result)) {
                return Optional.empty();
            }
            return Optional.of(result);
        }
        catch (Exception e) {
            LOG.error("An exception occurred while attempting to get a DynamoDb partition item for {}", (Object)sourcePartitionKey, (Object)e);
            return Optional.empty();
        }
    }

    public Optional<SourcePartitionStoreItem> getAvailablePartition(String ownerId, Duration ownershipTimeout, SourcePartitionStatus sourcePartitionStatus, String sourceStatusCombinationKey, int pageLimit, Duration ttl) {
        try {
            DynamoDbIndex sourceStatusIndex = this.table.index(SOURCE_STATUS_COMBINATION_KEY_GLOBAL_SECONDARY_INDEX);
            QueryEnhancedRequest queryEnhancedRequest = QueryEnhancedRequest.builder().limit(Integer.valueOf(pageLimit)).queryConditional(QueryConditional.keyEqualTo((Key)Key.builder().partitionValue(sourceStatusCombinationKey).build())).build();
            SdkIterable availableItems = sourceStatusIndex.query(queryEnhancedRequest);
            for (Page page : availableItems) {
                for (DynamoDbSourcePartitionItem item : page.items()) {
                    boolean acquired;
                    if (SourcePartitionStatus.ASSIGNED.equals((Object)sourcePartitionStatus) && item.getPartitionOwnershipTimeout() != null && Instant.now().isBefore(item.getPartitionOwnershipTimeout())) {
                        return Optional.empty();
                    }
                    if (SourcePartitionStatus.CLOSED.equals((Object)sourcePartitionStatus) && item.getReOpenAt() != null && Instant.now().isBefore(item.getReOpenAt())) {
                        return Optional.empty();
                    }
                    Instant partitionOwnershipTimeout = Instant.now().plus(ownershipTimeout);
                    item.setPartitionOwner(ownerId);
                    item.setPartitionOwnershipTimeout(partitionOwnershipTimeout);
                    item.setSourcePartitionStatus(SourcePartitionStatus.ASSIGNED);
                    item.setSourceStatusCombinationKey(String.format("%s|%s", item.getSourceIdentifier(), SourcePartitionStatus.ASSIGNED));
                    item.setPartitionPriority(partitionOwnershipTimeout.toString());
                    if (Objects.nonNull(ttl)) {
                        item.setExpirationTime(Instant.now().plus(ttl).getEpochSecond());
                    }
                    if (!(acquired = this.tryAcquirePartitionItem(item))) continue;
                    return Optional.of(item);
                }
            }
        }
        catch (Exception e) {
            LOG.error("An exception occurred while attempting to acquire a DynamoDb partition item for {}", (Object)sourceStatusCombinationKey, (Object)e);
            return Optional.empty();
        }
        return Optional.empty();
    }

    public List<SourcePartitionStoreItem> queryPartitionsByStatus(String sourceStatusCombinationKey, String partitionPriority) {
        ArrayList<SourcePartitionStoreItem> result = new ArrayList<SourcePartitionStoreItem>();
        try {
            DynamoDbIndex sourceStatusIndex = this.table.index(SOURCE_STATUS_COMBINATION_KEY_GLOBAL_SECONDARY_INDEX);
            QueryEnhancedRequest queryEnhancedRequest = QueryEnhancedRequest.builder().limit(Integer.valueOf(1000)).queryConditional(QueryConditional.sortGreaterThan((Key)Key.builder().partitionValue(sourceStatusCombinationKey).sortValue(partitionPriority).build())).build();
            SdkIterable availableItems = sourceStatusIndex.query(queryEnhancedRequest);
            for (Page page : availableItems) {
                for (DynamoDbSourcePartitionItem item : page.items()) {
                    result.add(item);
                }
            }
        }
        catch (Exception e) {
            LOG.error("An exception occurred while attempting to query partition items with {} due to {}", (Object)sourceStatusCombinationKey, (Object)e);
        }
        return result;
    }

    public List<SourcePartitionStoreItem> queryAllPartitions(String sourceIdentifier) {
        ArrayList<SourcePartitionStoreItem> result = new ArrayList<SourcePartitionStoreItem>();
        try {
            QueryEnhancedRequest queryEnhancedRequest = QueryEnhancedRequest.builder().limit(Integer.valueOf(1000)).queryConditional(QueryConditional.keyEqualTo((Key)Key.builder().partitionValue(sourceIdentifier).build())).build();
            PageIterable availableItems = this.table.query(queryEnhancedRequest);
            for (Page page : availableItems) {
                result.addAll(page.items());
            }
        }
        catch (Exception e) {
            LOG.error("An exception occurred while attempting to query all partition items with identifier {}", (Object)sourceIdentifier, (Object)e);
        }
        return result;
    }
}

