/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.dataprepper.plugins.sink.sqs;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.time.Instant;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.UUID;
import java.util.function.BiConsumer;
import org.opensearch.dataprepper.logging.DataPrepperMarkers;
import org.opensearch.dataprepper.model.codec.OutputCodec;
import org.opensearch.dataprepper.model.event.Event;
import org.opensearch.dataprepper.model.sink.OutputCodecContext;
import org.opensearch.dataprepper.plugins.accumulator.BufferFactory;
import org.opensearch.dataprepper.plugins.sink.sqs.SqsSinkBatchEntry;
import org.opensearch.dataprepper.plugins.sink.sqs.SqsSinkMetrics;
import org.opensearch.dataprepper.plugins.sink.sqs.SqsThresholdConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.services.sqs.SqsClient;
import software.amazon.awssdk.services.sqs.model.BatchResultErrorEntry;
import software.amazon.awssdk.services.sqs.model.RequestThrottledException;
import software.amazon.awssdk.services.sqs.model.SendMessageBatchRequest;
import software.amazon.awssdk.services.sqs.model.SendMessageBatchRequestEntry;
import software.amazon.awssdk.services.sqs.model.SendMessageBatchResponse;
import software.amazon.awssdk.services.sqs.model.SqsException;

public class SqsSinkBatch {
    private static final Logger LOG = LoggerFactory.getLogger(SqsSinkBatch.class);
    public static final int MAX_MESSAGES_PER_BATCH = 10;
    public static final int MAX_BATCH_SIZE_BYTES = 262144;
    private static final String SQS_FIFO_SUFFIX = ".fifo";
    private long lastFlushedTime;
    private Map<String, SqsSinkBatchEntry> entries;
    private boolean flushReady;
    private boolean fifoQueue;
    private String queueUrl;
    private final long maxMessageSize;
    private final int maxEvents;
    private final OutputCodecContext codecContext;
    private final OutputCodec codec;
    private final SqsClient sqsClient;
    private final BufferFactory bufferFactory;
    private final SqsSinkMetrics sinkMetrics;
    private String currentId;
    private SqsSinkBatchEntry currentBatchEntry;
    private final BiConsumer<SqsSinkBatchEntry, String> addToDLQList;

    public SqsSinkBatch(BufferFactory bufferFactory, SqsClient sqsClient, SqsSinkMetrics sinkMetrics, String queueUrl, OutputCodec codec, OutputCodecContext codecContext, SqsThresholdConfig thresholdConfig, BiConsumer<SqsSinkBatchEntry, String> addToDLQList) {
        this.bufferFactory = bufferFactory;
        this.maxMessageSize = thresholdConfig.getMaxMessageSizeBytes();
        this.maxEvents = thresholdConfig.getMaxEventsPerMessage();
        this.addToDLQList = addToDLQList;
        this.codec = codec;
        this.sinkMetrics = sinkMetrics;
        this.codecContext = codecContext;
        this.queueUrl = queueUrl;
        this.sqsClient = sqsClient;
        this.lastFlushedTime = Instant.now().toEpochMilli();
        this.flushReady = false;
        this.fifoQueue = queueUrl.endsWith(SQS_FIFO_SUFFIX);
        this.entries = new HashMap<String, SqsSinkBatchEntry>();
        this.currentBatchEntry = null;
        this.currentId = null;
    }

    public String getQueueUrl() {
        return this.queueUrl;
    }

    private boolean isFull() {
        return this.entries.size() == 10 && this.currentBatchEntry.getEventCount() == this.maxEvents;
    }

    public boolean willExceedLimits(long estimatedSize) {
        if (this.getCurrentBatchSize() + estimatedSize > 262144L) {
            return true;
        }
        if (this.currentBatchEntry != null && this.currentBatchEntry.getEventCount() < this.maxEvents && this.currentBatchEntry.getSize() + estimatedSize <= this.maxMessageSize) {
            return false;
        }
        return this.entries.size() == 10;
    }

    public boolean addEntry(Event event, String groupId, String deDupId, long estimatedSize) throws Exception {
        String id;
        if (this.currentBatchEntry != null) {
            if (this.currentBatchEntry.getEventCount() < this.maxEvents && this.currentBatchEntry.getSize() + estimatedSize < this.maxMessageSize) {
                this.currentBatchEntry.addEvent(event);
                return this.isFull();
            }
            try {
                this.currentBatchEntry.complete();
            }
            catch (IOException ex) {
                this.addToDLQList.accept(this.currentBatchEntry, ex.getMessage());
                this.entries.remove(this.currentId);
            }
        }
        if (this.entries.size() == 10) {
            throw new RuntimeException("Exceeds max messages per batch");
        }
        if (groupId == null) {
            groupId = UUID.randomUUID().toString();
        }
        if (deDupId == null) {
            deDupId = UUID.randomUUID().toString();
        }
        this.currentBatchEntry = new SqsSinkBatchEntry(this.bufferFactory.getBuffer(), groupId, deDupId, this.codec, this.codecContext);
        this.currentBatchEntry.addEvent(event);
        this.currentId = id = UUID.randomUUID().toString();
        this.entries.put(id, this.currentBatchEntry);
        return this.isFull();
    }

    public long getLastFlushedTime() {
        return this.lastFlushedTime;
    }

    public long getCurrentBatchSize() {
        long sum = 0L;
        for (Map.Entry<String, SqsSinkBatchEntry> entry : this.entries.entrySet()) {
            SqsSinkBatchEntry batchEntry = entry.getValue();
            sum += batchEntry.getSize();
            if (!this.fifoQueue) continue;
            if (batchEntry.getGroupId() != null) {
                sum += (long)batchEntry.getGroupId().length();
            }
            if (batchEntry.getDedupId() == null) continue;
            sum += (long)batchEntry.getDedupId().length();
        }
        return sum;
    }

    public int getEventCount() {
        return this.entries.values().stream().mapToInt(SqsSinkBatchEntry::getEventCount).sum();
    }

    public void setFlushReady() {
        Iterator<Map.Entry<String, SqsSinkBatchEntry>> iterator = this.entries.entrySet().iterator();
        while (iterator.hasNext()) {
            Map.Entry<String, SqsSinkBatchEntry> entry = iterator.next();
            try {
                entry.getValue().complete();
            }
            catch (IOException ex) {
                this.addToDLQList.accept(entry.getValue(), ex.getMessage());
                iterator.remove();
            }
        }
        this.flushReady = true;
    }

    public boolean isReady() {
        return this.flushReady;
    }

    private SendMessageBatchRequestEntry getRequestEntry(String id, SqsSinkBatchEntry entry) {
        SendMessageBatchRequestEntry.Builder builder = SendMessageBatchRequestEntry.builder().id(id).messageBody(entry.getBody());
        if (this.fifoQueue) {
            builder = builder.messageGroupId(entry.getGroupId()).messageDeduplicationId(entry.getDedupId());
        }
        return (SendMessageBatchRequestEntry)builder.build();
    }

    private boolean isRetryableException(SqsException e) {
        return e instanceof RequestThrottledException;
    }

    private long getEntrySize(SqsSinkBatchEntry entry) {
        long result = entry.getBody().getBytes(StandardCharsets.UTF_8).length;
        if (this.fifoQueue) {
            if (entry.getGroupId() != null) {
                result += (long)entry.getGroupId().getBytes(StandardCharsets.UTF_8).length;
            }
            if (entry.getDedupId() != null) {
                result += (long)entry.getDedupId().getBytes(StandardCharsets.UTF_8).length;
            }
        }
        return result;
    }

    public boolean flushOnce() {
        SendMessageBatchResponse flushResponse;
        if (!this.isReady() || this.entries.size() == 0) {
            return true;
        }
        ArrayList<SendMessageBatchRequestEntry> requestEntries = new ArrayList<SendMessageBatchRequestEntry>();
        long requestSize = 0L;
        for (Map.Entry<String, SqsSinkBatchEntry> groupEntry : this.entries.entrySet()) {
            Iterator<Map.Entry<String, SqsSinkBatchEntry>> id = groupEntry.getKey();
            SqsSinkBatchEntry sqsSinkBatchEntry = groupEntry.getValue();
            requestEntries.add(this.getRequestEntry((String)((Object)id), sqsSinkBatchEntry));
            requestSize += this.getEntrySize(sqsSinkBatchEntry);
        }
        SendMessageBatchRequest batchRequest = (SendMessageBatchRequest)SendMessageBatchRequest.builder().queueUrl(this.queueUrl).entries(requestEntries).build();
        try {
            flushResponse = this.sqsClient.sendMessageBatch(batchRequest);
            this.sinkMetrics.recordRequestSize(requestSize);
        }
        catch (SqsException e) {
            LOG.error(DataPrepperMarkers.NOISY, "Failed to send messages to SQS: {}", (Object)e.getMessage());
            this.sinkMetrics.incrementRequestsFailedCounter(1);
            this.sinkMetrics.incrementEventsFailedCounter(this.entries.size());
            if (!this.isRetryableException(e)) {
                for (Map.Entry<String, SqsSinkBatchEntry> entry : this.entries.entrySet()) {
                    this.addToDLQList.accept(entry.getValue(), e.getMessage());
                }
                this.entries.clear();
                Object flushResponse2 = null;
                return true;
            }
            return false;
        }
        this.sinkMetrics.incrementRequestsSuccessCounter(1);
        boolean flushResult = false;
        if (!flushResponse.hasFailed()) {
            for (SendMessageBatchRequestEntry sendMessageBatchRequestEntry : requestEntries) {
                SqsSinkBatchEntry sqsSinkBatchEntry = this.entries.get(sendMessageBatchRequestEntry.id());
                sqsSinkBatchEntry.releaseEventHandles(true);
                this.sinkMetrics.incrementEventsSuccessCounter(sqsSinkBatchEntry.getEventCount());
            }
            this.entries.clear();
        } else {
            HashMap<String, SqsSinkBatchEntry> newEntries = new HashMap<String, SqsSinkBatchEntry>();
            this.sinkMetrics.incrementEventsFailedCounter(flushResponse.failed().size());
            for (BatchResultErrorEntry batchResultErrorEntry : flushResponse.failed()) {
                SqsSinkBatchEntry batchEntry = this.entries.get(batchResultErrorEntry.id());
                if (!batchResultErrorEntry.senderFault().booleanValue()) {
                    newEntries.put(batchResultErrorEntry.id(), batchEntry);
                } else {
                    this.addToDLQList.accept(batchEntry, batchResultErrorEntry.message());
                }
                this.entries.remove(batchResultErrorEntry.id());
            }
            this.sinkMetrics.incrementEventsSuccessCounter(this.entries.size());
            for (Map.Entry<String, SqsSinkBatchEntry> entry : this.entries.entrySet()) {
                entry.getValue().releaseEventHandles(true);
            }
            this.entries.clear();
            this.entries = newEntries;
        }
        this.lastFlushedTime = Instant.now().toEpochMilli();
        return this.entries.size() == 0;
    }

    public Map<String, SqsSinkBatchEntry> getEntries() {
        return this.entries;
    }
}

