/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.dataprepper.plugins.sink.otlp.http;

import com.google.common.annotations.VisibleForTesting;
import com.linecorp.armeria.client.WebClient;
import com.linecorp.armeria.client.retry.Backoff;
import com.linecorp.armeria.client.retry.RetryRuleWithContent;
import com.linecorp.armeria.client.retry.RetryRuleWithContentBuilder;
import com.linecorp.armeria.client.retry.RetryingClient;
import com.linecorp.armeria.client.retry.RetryingClientBuilder;
import com.linecorp.armeria.common.HttpData;
import com.linecorp.armeria.common.HttpMethod;
import com.linecorp.armeria.common.HttpRequest;
import com.linecorp.armeria.common.RequestHeaders;
import com.linecorp.armeria.common.RequestHeadersBuilder;
import io.opentelemetry.proto.collector.trace.v1.ExportTracePartialSuccess;
import io.opentelemetry.proto.collector.trace.v1.ExportTraceServiceRequest;
import io.opentelemetry.proto.collector.trace.v1.ExportTraceServiceResponse;
import io.opentelemetry.proto.trace.v1.ResourceSpans;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import java.util.stream.Collectors;
import javax.annotation.Nonnull;
import org.opensearch.dataprepper.aws.api.AwsCredentialsSupplier;
import org.opensearch.dataprepper.model.event.EventHandle;
import org.opensearch.dataprepper.plugins.sink.otlp.configuration.OtlpSinkConfig;
import org.opensearch.dataprepper.plugins.sink.otlp.http.GzipCompressor;
import org.opensearch.dataprepper.plugins.sink.otlp.http.SigV4Signer;
import org.opensearch.dataprepper.plugins.sink.otlp.metrics.OtlpSinkMetrics;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.http.SdkHttpFullRequest;
import software.amazon.awssdk.utils.Pair;

public class OtlpHttpSender {
    private static final Logger LOG = LoggerFactory.getLogger(OtlpHttpSender.class);
    private static final Set<Integer> RETRYABLE_STATUS_CODES = Set.of(Integer.valueOf(429), Integer.valueOf(502), Integer.valueOf(503), Integer.valueOf(504));
    private final SigV4Signer signer;
    private final WebClient webClient;
    private final OtlpSinkMetrics sinkMetrics;
    private final Function<byte[], byte[]> gzipCompressor;

    public OtlpHttpSender(@Nonnull AwsCredentialsSupplier awsCredentialsSupplier, @Nonnull OtlpSinkConfig config, @Nonnull OtlpSinkMetrics sinkMetrics) {
        this(sinkMetrics, new GzipCompressor(sinkMetrics), new SigV4Signer(awsCredentialsSupplier, config), OtlpHttpSender.buildWebClient(config));
    }

    @VisibleForTesting
    OtlpHttpSender(@Nonnull OtlpSinkMetrics sinkMetrics, @Nonnull Function<byte[], byte[]> gzipCompressor, SigV4Signer signer, WebClient webClient) {
        this.sinkMetrics = sinkMetrics;
        this.gzipCompressor = gzipCompressor;
        this.signer = signer;
        this.webClient = webClient;
    }

    private static WebClient buildWebClient(OtlpSinkConfig config) {
        RetryRuleWithContent retryRule = ((RetryRuleWithContentBuilder)RetryRuleWithContent.builder().onStatus((ctx, status) -> RETRYABLE_STATUS_CODES.contains(status.code()))).thenBackoff(Backoff.exponential((long)100L, (long)10000L).withJitter(0.2));
        long estimatedContentLimit = Math.max(1L, config.getMaxBatchSize()) * (long)(config.getMaxRetries() + 1);
        int safeContentLimit = (int)Math.min(estimatedContentLimit, Integer.MAX_VALUE);
        RetryingClientBuilder retryingClientBuilder = RetryingClient.builder((RetryRuleWithContent)retryRule, (int)safeContentLimit).maxTotalAttempts(config.getMaxRetries() + 1);
        long httpTimeoutMs = Math.min(Math.max(config.getFlushTimeoutMillis() * 2L, 3000L), 10000L);
        return WebClient.builder().decorator(retryingClientBuilder.newDecorator()).responseTimeoutMillis(httpTimeoutMs).maxResponseLength((long)safeContentLimit).build();
    }

    public void send(@Nonnull List<Pair<ResourceSpans, EventHandle>> batch) {
        if (batch.isEmpty()) {
            return;
        }
        List<Pair<ResourceSpans, EventHandle>> immutableBatch = List.copyOf(batch);
        Pair<byte[], byte[]> payloadAndCompressedPayload = this.getPayloadAndCompressedPayload(immutableBatch);
        int spans = immutableBatch.size();
        if (((byte[])payloadAndCompressedPayload.right()).length == 0) {
            this.sinkMetrics.incrementFailedSpansCount(spans);
            this.releaseAllEventHandle(immutableBatch, false);
            return;
        }
        HttpRequest request = this.buildHttpRequest((byte[])payloadAndCompressedPayload.right());
        long startTime = System.currentTimeMillis();
        ((CompletableFuture)this.webClient.execute(request).aggregate().thenAccept(response -> {
            long latency = System.currentTimeMillis() - startTime;
            this.sinkMetrics.recordHttpLatency(latency);
            this.sinkMetrics.incrementPayloadSize(((byte[])payloadAndCompressedPayload.left()).length);
            this.sinkMetrics.incrementPayloadGzipSize(((byte[])payloadAndCompressedPayload.right()).length);
            int statusCode = response.status().code();
            byte[] responseBytes = response.content().array();
            this.handleResponse(statusCode, responseBytes, immutableBatch);
        })).exceptionally(e -> {
            LOG.error("Failed to send {} spans.", (Object)spans, e);
            this.sinkMetrics.incrementRejectedSpansCount(spans);
            this.releaseAllEventHandle(immutableBatch, false);
            return null;
        });
    }

    private Pair<byte[], byte[]> getPayloadAndCompressedPayload(List<Pair<ResourceSpans, EventHandle>> batch) {
        ExportTraceServiceRequest request = ExportTraceServiceRequest.newBuilder().addAllResourceSpans((Iterable)batch.stream().map(Pair::left).collect(Collectors.toList())).build();
        byte[] payload = request.toByteArray();
        byte[] compressedPayload = this.gzipCompressor.apply(payload);
        return Pair.of((Object)payload, (Object)compressedPayload);
    }

    private HttpRequest buildHttpRequest(byte[] compressedPayload) {
        SdkHttpFullRequest signedRequest = this.signer.signRequest(compressedPayload);
        RequestHeadersBuilder headersBuilder = RequestHeaders.builder().method(HttpMethod.POST).scheme(signedRequest.getUri().getScheme()).path(signedRequest.getUri().getRawPath()).authority(signedRequest.getUri().getAuthority());
        signedRequest.headers().forEach((k, vList) -> vList.forEach(v -> headersBuilder.add((CharSequence)k, v)));
        return HttpRequest.of((RequestHeaders)headersBuilder.build(), (HttpData)HttpData.wrap((byte[])compressedPayload));
    }

    private void handleResponse(int statusCode, byte[] responseBytes, List<Pair<ResourceSpans, EventHandle>> batch) {
        this.sinkMetrics.recordResponseCode(statusCode);
        if (statusCode >= 200 && statusCode < 300) {
            this.handleSuccessfulResponse(responseBytes, batch);
            return;
        }
        String responseBody = responseBytes != null ? new String(responseBytes, StandardCharsets.UTF_8) : "<no body>";
        LOG.error("Non-successful OTLP response. Status: {}, Response: {}", (Object)statusCode, (Object)responseBody);
        this.sinkMetrics.incrementRejectedSpansCount(batch.size());
        this.releaseAllEventHandle(batch, false);
    }

    private void handleSuccessfulResponse(byte[] responseBytes, List<Pair<ResourceSpans, EventHandle>> batch) {
        int spans = batch.size();
        if (responseBytes == null) {
            this.sinkMetrics.incrementRecordsOut(spans);
            this.releaseAllEventHandle(batch, true);
            return;
        }
        try {
            ExportTraceServiceResponse otlpResponse = ExportTraceServiceResponse.parseFrom((byte[])responseBytes);
            if (otlpResponse.hasPartialSuccess()) {
                ExportTracePartialSuccess partial = otlpResponse.getPartialSuccess();
                long rejectedSpans = partial.getRejectedSpans();
                String errorMessage = partial.getErrorMessage();
                if (rejectedSpans > 0L) {
                    LOG.error("OTLP Partial Success: rejectedSpans={}, message={}", (Object)rejectedSpans, (Object)errorMessage);
                    this.sinkMetrics.incrementRejectedSpansCount(rejectedSpans);
                }
                long deliveredSpans = (long)spans - rejectedSpans;
                this.sinkMetrics.incrementRecordsOut(deliveredSpans);
                this.releaseAllEventHandle(batch, true);
            } else {
                this.sinkMetrics.incrementRecordsOut(spans);
                this.releaseAllEventHandle(batch, true);
            }
        }
        catch (Exception e) {
            LOG.error("Could not parse OTLP response as ExportTraceServiceResponse: {}", (Object)e.getMessage());
            this.sinkMetrics.incrementErrorsCount();
            this.sinkMetrics.incrementRecordsOut(spans);
            this.releaseAllEventHandle(batch, true);
        }
    }

    private void releaseAllEventHandle(@Nonnull List<Pair<ResourceSpans, EventHandle>> batch, boolean success) {
        batch.forEach(pair -> ((EventHandle)pair.right()).release(success));
    }
}

