/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.dataprepper.plugins.kinesis.source.processor;

import com.google.common.annotations.VisibleForTesting;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.DistributionSummary;
import java.time.Duration;
import java.util.List;
import java.util.ListIterator;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;
import org.opensearch.dataprepper.buffer.common.BufferAccumulator;
import org.opensearch.dataprepper.common.concurrent.BackgroundThreadFactory;
import org.opensearch.dataprepper.metrics.PluginMetrics;
import org.opensearch.dataprepper.model.acknowledgements.AcknowledgementSet;
import org.opensearch.dataprepper.model.acknowledgements.AcknowledgementSetManager;
import org.opensearch.dataprepper.model.event.Event;
import org.opensearch.dataprepper.model.record.Record;
import org.opensearch.dataprepper.plugins.kinesis.source.configuration.KinesisSourceConfig;
import org.opensearch.dataprepper.plugins.kinesis.source.configuration.KinesisStreamConfig;
import org.opensearch.dataprepper.plugins.kinesis.source.converter.KinesisRecordConverter;
import org.opensearch.dataprepper.plugins.kinesis.source.exceptions.KinesisStreamNotFoundException;
import org.opensearch.dataprepper.plugins.kinesis.source.processor.KinesisCheckpointerRecord;
import org.opensearch.dataprepper.plugins.kinesis.source.processor.KinesisCheckpointerTracker;
import org.opensearch.dataprepper.plugins.kinesis.source.processor.KinesisInputOutputRecord;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.arns.Arn;
import software.amazon.kinesis.common.StreamIdentifier;
import software.amazon.kinesis.exceptions.InvalidStateException;
import software.amazon.kinesis.exceptions.ShutdownException;
import software.amazon.kinesis.exceptions.ThrottlingException;
import software.amazon.kinesis.lifecycle.events.InitializationInput;
import software.amazon.kinesis.lifecycle.events.LeaseLostInput;
import software.amazon.kinesis.lifecycle.events.ProcessRecordsInput;
import software.amazon.kinesis.lifecycle.events.ShardEndedInput;
import software.amazon.kinesis.lifecycle.events.ShutdownRequestedInput;
import software.amazon.kinesis.processor.RecordProcessorCheckpointer;
import software.amazon.kinesis.processor.ShardRecordProcessor;
import software.amazon.kinesis.retrieval.KinesisClientRecord;
import software.amazon.kinesis.retrieval.kpl.ExtendedSequenceNumber;

public class KinesisRecordProcessor
implements ShardRecordProcessor {
    private static final Logger LOG = LoggerFactory.getLogger(KinesisRecordProcessor.class);
    private static final int DEFAULT_MONITOR_WAIT_TIME_MS = 15000;
    private static final Duration ACKNOWLEDGEMENT_SET_TIMEOUT = Duration.ofSeconds(20L);
    private final StreamIdentifier streamIdentifier;
    private final KinesisStreamConfig kinesisStreamConfig;
    private final Duration checkpointInterval;
    private final KinesisSourceConfig kinesisSourceConfig;
    private final BufferAccumulator<Record<Event>> bufferAccumulator;
    private final KinesisRecordConverter kinesisRecordConverter;
    private final KinesisCheckpointerTracker kinesisCheckpointerTracker;
    private final ExecutorService executorService;
    private String kinesisShardId;
    private long lastCheckpointTimeInMillis;
    private final int bufferTimeoutMillis;
    private final AcknowledgementSetManager acknowledgementSetManager;
    private final Counter acknowledgementSetSuccesses;
    private final Counter acknowledgementSetFailures;
    private final Counter recordsProcessed;
    private final Counter recordProcessingErrors;
    private final Counter checkpointFailures;
    private final DistributionSummary bytesReceivedSummary;
    private final DistributionSummary bytesProcessedSummary;
    public static final String ACKNOWLEDGEMENT_SET_SUCCESS_METRIC_NAME = "acknowledgementSetSuccesses";
    public static final String ACKNOWLEDGEMENT_SET_FAILURES_METRIC_NAME = "acknowledgementSetFailures";
    public static final String KINESIS_RECORD_PROCESSED_METRIC_NAME = "recordProcessed";
    public static final String KINESIS_RECORD_PROCESSING_ERRORS_METRIC_NAME = "recordProcessingErrors";
    public static final String KINESIS_RECORD_BYTES_RECEIVED_METRIC_NAME = "bytesReceived";
    public static final String KINESIS_RECORD_BYTES_PROCESSED_METRIC_NAME = "bytesProcessed";
    public static final String KINESIS_CHECKPOINT_FAILURES = "checkpointFailures";
    public static final String KINESIS_STREAM_TAG_KEY = "stream";
    private AtomicBoolean isStopRequested;

    public KinesisRecordProcessor(BufferAccumulator<Record<Event>> bufferAccumulator, KinesisSourceConfig kinesisSourceConfig, AcknowledgementSetManager acknowledgementSetManager, PluginMetrics pluginMetrics, KinesisRecordConverter kinesisRecordConverter, KinesisCheckpointerTracker kinesisCheckpointerTracker, StreamIdentifier streamIdentifier) {
        this.bufferTimeoutMillis = (int)kinesisSourceConfig.getBufferTimeout().toMillis();
        this.streamIdentifier = streamIdentifier;
        this.kinesisSourceConfig = kinesisSourceConfig;
        this.kinesisStreamConfig = this.getStreamConfig(kinesisSourceConfig);
        this.kinesisRecordConverter = kinesisRecordConverter;
        this.acknowledgementSetManager = acknowledgementSetManager;
        this.acknowledgementSetSuccesses = pluginMetrics.counterWithTags(ACKNOWLEDGEMENT_SET_SUCCESS_METRIC_NAME, new String[]{KINESIS_STREAM_TAG_KEY, streamIdentifier.streamName()});
        this.acknowledgementSetFailures = pluginMetrics.counterWithTags(ACKNOWLEDGEMENT_SET_FAILURES_METRIC_NAME, new String[]{KINESIS_STREAM_TAG_KEY, streamIdentifier.streamName()});
        this.recordsProcessed = pluginMetrics.counterWithTags(KINESIS_RECORD_PROCESSED_METRIC_NAME, new String[]{KINESIS_STREAM_TAG_KEY, streamIdentifier.streamName()});
        this.recordProcessingErrors = pluginMetrics.counterWithTags(KINESIS_RECORD_PROCESSING_ERRORS_METRIC_NAME, new String[]{KINESIS_STREAM_TAG_KEY, streamIdentifier.streamName()});
        this.checkpointFailures = pluginMetrics.counterWithTags(KINESIS_CHECKPOINT_FAILURES, new String[]{KINESIS_STREAM_TAG_KEY, streamIdentifier.streamName()});
        this.bytesReceivedSummary = pluginMetrics.summary(KINESIS_RECORD_BYTES_RECEIVED_METRIC_NAME);
        this.bytesProcessedSummary = pluginMetrics.summary(KINESIS_RECORD_BYTES_PROCESSED_METRIC_NAME);
        this.checkpointInterval = this.kinesisStreamConfig.getCheckPointInterval();
        this.bufferAccumulator = bufferAccumulator;
        this.kinesisCheckpointerTracker = kinesisCheckpointerTracker;
        this.executorService = Executors.newSingleThreadExecutor((ThreadFactory)BackgroundThreadFactory.defaultExecutorThreadFactory((String)"kinesis-ack-monitor"));
        this.isStopRequested = new AtomicBoolean(false);
    }

    private KinesisStreamConfig getStreamConfig(KinesisSourceConfig kinesisSourceConfig) {
        List matchedStreamConfigs = kinesisSourceConfig.getStreams().stream().filter(streamConfig -> {
            if (this.streamIdentifier.streamArnOptional().isPresent()) {
                Arn streamIdentifierArn = (Arn)this.streamIdentifier.streamArnOptional().get();
                Arn streamConfigArn = Arn.fromString((String)streamConfig.getStreamArn());
                return streamIdentifierArn.equals((Object)streamConfigArn);
            }
            if (Objects.nonNull(streamConfig.getStreamArn())) {
                String streamName = Arn.fromString((String)streamConfig.getStreamArn()).resource().resource();
                return streamName.equals(this.streamIdentifier.streamName());
            }
            return streamConfig.getName().equals(this.streamIdentifier.streamName());
        }).collect(Collectors.toList());
        if (matchedStreamConfigs.isEmpty()) {
            throw new KinesisStreamNotFoundException(String.format("Kinesis stream not found for %s", this.streamIdentifier.streamName()));
        }
        if (matchedStreamConfigs.size() > 1) {
            throw new IllegalStateException(String.format("Multiple stream configurations found for %s. Expected exactly one.", this.streamIdentifier.streamName()));
        }
        return (KinesisStreamConfig)matchedStreamConfigs.get(0);
    }

    public void initialize(InitializationInput initializationInput) {
        this.kinesisShardId = initializationInput.shardId();
        String kinesisStreamName = this.streamIdentifier.streamName();
        LOG.info("Initialize Processor for stream: {},  shard: {}", (Object)kinesisStreamName, (Object)this.kinesisShardId);
        this.lastCheckpointTimeInMillis = System.currentTimeMillis();
        if (this.kinesisSourceConfig.isAcknowledgments()) {
            this.executorService.submit(() -> this.monitorCheckpoint(this.executorService));
        }
    }

    private void monitorCheckpoint(ExecutorService executorService) {
        while (!this.isStopRequested.get()) {
            if (System.currentTimeMillis() - this.lastCheckpointTimeInMillis >= this.checkpointInterval.toMillis()) {
                this.doCheckpoint();
            }
            try {
                Thread.sleep(15000L);
            }
            catch (InterruptedException ex) {
                // empty catch block
                break;
            }
        }
        executorService.shutdown();
    }

    private AcknowledgementSet createAcknowledgmentSet(ProcessRecordsInput processRecordsInput, ExtendedSequenceNumber extendedSequenceNumber) {
        return this.acknowledgementSetManager.create(result -> {
            String kinesisStreamName = this.streamIdentifier.streamName();
            if (result.booleanValue()) {
                this.acknowledgementSetSuccesses.increment();
                this.kinesisCheckpointerTracker.markSequenceNumberForCheckpoint(extendedSequenceNumber);
                LOG.debug("acknowledgements received for stream: {}, shardId: {}", (Object)kinesisStreamName, (Object)this.kinesisShardId);
            } else {
                this.acknowledgementSetFailures.increment();
                LOG.debug("acknowledgements received with false for stream: {}, shardId: {}", (Object)kinesisStreamName, (Object)this.kinesisShardId);
            }
        }, ACKNOWLEDGEMENT_SET_TIMEOUT);
    }

    public void processRecords(ProcessRecordsInput processRecordsInput) {
        try {
            Optional<Object> acknowledgementSetOpt = Optional.empty();
            boolean acknowledgementsEnabled = this.kinesisSourceConfig.isAcknowledgments();
            ExtendedSequenceNumber extendedSequenceNumber = this.getLatestSequenceNumberFromInput(processRecordsInput);
            if (acknowledgementsEnabled) {
                acknowledgementSetOpt = Optional.of(this.createAcknowledgmentSet(processRecordsInput, extendedSequenceNumber));
            }
            this.kinesisCheckpointerTracker.addRecordForCheckpoint(extendedSequenceNumber, processRecordsInput.checkpointer());
            List<KinesisInputOutputRecord> kinesisOutputRecords = this.kinesisRecordConverter.convert(this.kinesisStreamConfig.getCompression().getDecompressionEngine(), processRecordsInput.records(), this.streamIdentifier.streamName());
            int eventCount = 0;
            for (KinesisInputOutputRecord kinesisInputOutputRecord : kinesisOutputRecords) {
                Record<Event> dataPrepperRecord = kinesisInputOutputRecord.getDataPrepperRecord();
                long incomingRecordSizeBytes = kinesisInputOutputRecord.getIncomingRecordSizeBytes();
                this.bytesReceivedSummary.record((double)incomingRecordSizeBytes);
                Event event = (Event)dataPrepperRecord.getData();
                acknowledgementSetOpt.ifPresent(acknowledgementSet -> acknowledgementSet.add(event));
                this.bufferAccumulator.add(dataPrepperRecord);
                this.bytesProcessedSummary.record((double)incomingRecordSizeBytes);
                ++eventCount;
            }
            this.bufferAccumulator.flush();
            this.recordsProcessed.increment((double)eventCount);
            if (!acknowledgementsEnabled) {
                this.kinesisCheckpointerTracker.markSequenceNumberForCheckpoint(extendedSequenceNumber);
            }
            LOG.debug("Number of Records {} written for stream: {}, shardId: {}", new Object[]{eventCount, this.streamIdentifier.streamName(), this.kinesisShardId});
            acknowledgementSetOpt.ifPresent(AcknowledgementSet::complete);
            if (!acknowledgementsEnabled && System.currentTimeMillis() - this.lastCheckpointTimeInMillis >= this.checkpointInterval.toMillis()) {
                this.doCheckpoint();
            }
        }
        catch (Exception ex) {
            this.recordProcessingErrors.increment();
            LOG.error("Failed writing shard data to buffer: ", (Throwable)ex);
        }
    }

    public void leaseLost(LeaseLostInput leaseLostInput) {
        LOG.debug("Lease Lost");
    }

    public void shardEnded(ShardEndedInput shardEndedInput) {
        String kinesisStream = this.streamIdentifier.streamName();
        LOG.debug("Reached shard end, checkpointing for stream: {}, shardId: {}", (Object)kinesisStream, (Object)this.kinesisShardId);
        this.checkpoint(shardEndedInput.checkpointer());
    }

    public void shutdownRequested(ShutdownRequestedInput shutdownRequestedInput) {
        String kinesisStream = this.streamIdentifier.streamName();
        this.isStopRequested.set(true);
        LOG.debug("Scheduler is shutting down, checkpointing for stream: {}, shardId: {}", (Object)kinesisStream, (Object)this.kinesisShardId);
        this.checkpoint(shutdownRequestedInput.checkpointer());
    }

    @VisibleForTesting
    public void checkpoint(RecordProcessorCheckpointer checkpointer, String sequenceNumber, long subSequenceNumber) {
        try {
            String kinesisStream = this.streamIdentifier.streamName();
            LOG.debug("Checkpoint for stream: {}, shardId: {}, sequence: {}, subsequence: {}", new Object[]{kinesisStream, this.kinesisShardId, sequenceNumber, subSequenceNumber});
            checkpointer.checkpoint(sequenceNumber, subSequenceNumber);
        }
        catch (InvalidStateException | ShutdownException | ThrottlingException ex) {
            LOG.debug("Caught exception at checkpoint, skipping checkpoint.", ex);
            this.checkpointFailures.increment();
        }
    }

    private void doCheckpoint() {
        LOG.debug("Regular checkpointing for shard {}", (Object)this.kinesisShardId);
        Optional<KinesisCheckpointerRecord> kinesisCheckpointerRecordOptional = this.kinesisCheckpointerTracker.popLatestReadyToCheckpointRecord();
        if (kinesisCheckpointerRecordOptional.isPresent()) {
            ExtendedSequenceNumber lastExtendedSequenceNumber = kinesisCheckpointerRecordOptional.get().getExtendedSequenceNumber();
            RecordProcessorCheckpointer recordProcessorCheckpointer = kinesisCheckpointerRecordOptional.get().getCheckpointer();
            this.checkpoint(recordProcessorCheckpointer, lastExtendedSequenceNumber.sequenceNumber(), lastExtendedSequenceNumber.subSequenceNumber());
            this.lastCheckpointTimeInMillis = System.currentTimeMillis();
        }
    }

    private void checkpoint(RecordProcessorCheckpointer checkpointer) {
        try {
            String kinesisStream = this.streamIdentifier.streamName();
            LOG.debug("Checkpoint for stream: {}, shardId: {}", (Object)kinesisStream, (Object)this.kinesisShardId);
            checkpointer.checkpoint();
        }
        catch (InvalidStateException | ShutdownException | ThrottlingException ex) {
            LOG.debug("Caught exception at checkpoint, skipping checkpoint.", ex);
            this.checkpointFailures.increment();
        }
    }

    private ExtendedSequenceNumber getLatestSequenceNumberFromInput(ProcessRecordsInput processRecordsInput) {
        ListIterator recordIterator = processRecordsInput.records().listIterator();
        ExtendedSequenceNumber largestExtendedSequenceNumber = null;
        while (recordIterator.hasNext()) {
            KinesisClientRecord record = (KinesisClientRecord)recordIterator.next();
            ExtendedSequenceNumber extendedSequenceNumber = new ExtendedSequenceNumber(record.sequenceNumber(), Long.valueOf(record.subSequenceNumber()));
            if (largestExtendedSequenceNumber != null && largestExtendedSequenceNumber.compareTo(extendedSequenceNumber) >= 0) continue;
            largestExtendedSequenceNumber = extendedSequenceNumber;
        }
        return largestExtendedSequenceNumber;
    }
}

