/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.dataprepper.plugins.mongo.export;

import com.mongodb.MongoClientException;
import com.mongodb.client.MongoClient;
import com.mongodb.client.MongoCollection;
import com.mongodb.client.MongoCursor;
import com.mongodb.client.MongoDatabase;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.DistributionSummary;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import org.bson.BsonDocument;
import org.bson.Document;
import org.bson.conversions.Bson;
import org.opensearch.dataprepper.metrics.PluginMetrics;
import org.opensearch.dataprepper.model.acknowledgements.AcknowledgementSet;
import org.opensearch.dataprepper.model.event.Event;
import org.opensearch.dataprepper.plugins.mongo.buffer.RecordBufferWriter;
import org.opensearch.dataprepper.plugins.mongo.client.BsonHelper;
import org.opensearch.dataprepper.plugins.mongo.client.MongoDBConnection;
import org.opensearch.dataprepper.plugins.mongo.configuration.MongoDBSourceConfig;
import org.opensearch.dataprepper.plugins.mongo.converter.PartitionKeyRecordConverter;
import org.opensearch.dataprepper.plugins.mongo.coordination.partition.DataQueryPartition;
import org.opensearch.dataprepper.plugins.mongo.export.DataQueryPartitionCheckpoint;
import org.opensearch.dataprepper.plugins.mongo.model.S3PartitionStatus;
import org.opensearch.dataprepper.plugins.mongo.utils.DocumentDBSourceAggregateMetrics;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ExportPartitionWorker
implements Runnable {
    private static final Logger LOG = LoggerFactory.getLogger(ExportPartitionWorker.class);
    private static final int PARTITION_KEY_PARTS = 4;
    static final Duration VERSION_OVERLAP_TIME_FOR_EXPORT = Duration.ofMinutes(5L);
    static final String EXPORT_RECORDS_TOTAL_COUNT = "exportRecordsTotal";
    static final String SUCCESS_ITEM_COUNTER_NAME = "exportRecordsProcessed";
    static final String FAILURE_ITEM_COUNTER_NAME = "exportRecordProcessingErrors";
    static final String BYTES_RECEIVED = "bytesReceived";
    static final String BYTES_PROCESSED = "bytesProcessed";
    private static final String PARTITION_KEY_SPLITTER = "\\|";
    private static final String COLLECTION_SPLITTER = "\\.";
    private static final int DEFAULT_BATCH_SIZE = 100;
    private final int startLine;
    private static final int DEFAULT_CHECKPOINT_INTERVAL_MILLS = 120000;
    private static final int DEFAULT_PARTITION_CREATE_WAIT_INTERVAL_MILLIS = 60000;
    private static volatile boolean shouldStop = false;
    private final MongoDBSourceConfig sourceConfig;
    private final Counter exportRecordTotalCounter;
    private final Counter successItemsCounter;
    private final Counter failureItemsCounter;
    private final DistributionSummary bytesReceivedSummary;
    private final DistributionSummary bytesProcessedSummary;
    private final RecordBufferWriter recordBufferWriter;
    private final PartitionKeyRecordConverter recordConverter;
    private final DataQueryPartition dataQueryPartition;
    private final AcknowledgementSet acknowledgementSet;
    private final long exportStartTimeEpochMillis;
    private final DataQueryPartitionCheckpoint partitionCheckpoint;
    private final DocumentDBSourceAggregateMetrics documentDBAggregateMetrics;
    Optional<S3PartitionStatus> s3PartitionStatus = Optional.empty();

    public ExportPartitionWorker(RecordBufferWriter recordBufferWriter, PartitionKeyRecordConverter recordConverter, DataQueryPartition dataQueryPartition, AcknowledgementSet acknowledgementSet, MongoDBSourceConfig sourceConfig, DataQueryPartitionCheckpoint partitionCheckpoint, long exportStartTimeEpochMillis, PluginMetrics pluginMetrics, DocumentDBSourceAggregateMetrics documentDBAggregateMetrics) {
        this.recordBufferWriter = recordBufferWriter;
        this.recordConverter = recordConverter;
        this.dataQueryPartition = dataQueryPartition;
        this.acknowledgementSet = acknowledgementSet;
        this.sourceConfig = sourceConfig;
        this.partitionCheckpoint = partitionCheckpoint;
        this.startLine = 0;
        this.exportRecordTotalCounter = pluginMetrics.counter(EXPORT_RECORDS_TOTAL_COUNT);
        this.exportStartTimeEpochMillis = exportStartTimeEpochMillis;
        this.successItemsCounter = pluginMetrics.counter(SUCCESS_ITEM_COUNTER_NAME);
        this.failureItemsCounter = pluginMetrics.counter(FAILURE_ITEM_COUNTER_NAME);
        this.bytesReceivedSummary = pluginMetrics.summary(BYTES_RECEIVED);
        this.bytesProcessedSummary = pluginMetrics.summary(BYTES_PROCESSED);
        this.documentDBAggregateMetrics = documentDBAggregateMetrics;
    }

    private boolean shouldWaitForS3Partition(String collection) {
        this.s3PartitionStatus = this.partitionCheckpoint.getGlobalS3FolderCreationStatus(collection);
        return this.s3PartitionStatus.isEmpty();
    }

    @Override
    public void run() {
        List<String> s3Partitions;
        this.documentDBAggregateMetrics.getExportApiInvocations().increment();
        List<String> partitionKeys = List.of(this.dataQueryPartition.getPartitionKey().split(PARTITION_KEY_SPLITTER));
        if (partitionKeys.size() < 4) {
            this.documentDBAggregateMetrics.getExport5xxErrors().increment();
            throw new RuntimeException("Invalid Partition Key. Must as db.collection|gte|lte format. Key: " + this.dataQueryPartition.getPartitionKey());
        }
        List<String> collection = List.of(partitionKeys.get(0).split(COLLECTION_SPLITTER));
        String gte = partitionKeys.get(1);
        String lte = partitionKeys.get(2);
        String gteClassName = partitionKeys.get(3);
        String lteClassName = partitionKeys.get(4);
        if (collection.size() < 2) {
            this.documentDBAggregateMetrics.getExport4xxErrors().increment();
            throw new IllegalArgumentException("Invalid Collection Name. Must as db.collection format");
        }
        long lastCheckpointTime = System.currentTimeMillis();
        while (this.shouldWaitForS3Partition(this.dataQueryPartition.getCollection()) && !Thread.currentThread().isInterrupted()) {
            LOG.info("S3 partition was not complete for collection {}, waiting for partitions to be created before resuming export.", (Object)this.dataQueryPartition.getCollection());
            try {
                Thread.sleep(60000L);
            }
            catch (InterruptedException ex) {
                LOG.info("The ExportPartitionWorker was interrupted while waiting to retry, stopping the worker");
                Thread.currentThread().interrupt();
                break;
            }
        }
        if ((s3Partitions = this.s3PartitionStatus.get().getPartitions()).isEmpty()) {
            this.documentDBAggregateMetrics.getExport5xxErrors().increment();
            throw new IllegalStateException("S3 partitions are not created. Please check the S3 partition creator thread.");
        }
        this.recordConverter.initializePartitions(s3Partitions);
        try (MongoClient mongoClient = MongoDBConnection.getMongoClient(this.sourceConfig);){
            MongoDatabase db = mongoClient.getDatabase(collection.get(0));
            MongoCollection col = db.getCollection(partitionKeys.get(0).substring(collection.get(0).length() + 1));
            Bson query = BsonHelper.buildQuery(gte, lte, gteClassName, lteClassName);
            long totalRecords = 0L;
            long successRecords = 0L;
            long failedRecords = 0L;
            int recordCount = 0;
            int lastRecordNumberProcessed = 0;
            ArrayList<Event> records = new ArrayList<Event>();
            ArrayList<Long> recordBytes = new ArrayList<Long>();
            try (MongoCursor cursor = col.find(query).iterator();){
                while (cursor.hasNext() && !Thread.currentThread().isInterrupted()) {
                    if (shouldStop) {
                        this.partitionCheckpoint.checkpoint(lastRecordNumberProcessed);
                        LOG.warn("Loading data query {} was interrupted by a shutdown signal, giving up ownership of query partition", (Object)query);
                        throw new RuntimeException("Loading data query interrupted");
                    }
                    ++recordCount;
                    if (++totalRecords <= (long)this.startLine) continue;
                    this.exportRecordTotalCounter.increment();
                    try {
                        Document document = (Document)cursor.next();
                        String record = document.toJson(BsonHelper.JSON_WRITER_SETTINGS);
                        long bytes = record.getBytes().length;
                        recordBytes.add(bytes);
                        this.bytesReceivedSummary.record((double)bytes);
                        Optional<BsonDocument> primaryKeyDoc = Optional.ofNullable(document.toBsonDocument());
                        String primaryKeyBsonType = primaryKeyDoc.map(bsonDocument -> bsonDocument.get((Object)"_id").getBsonType().name()).orElse("UNKNOWN");
                        long eventVersionNumber = (this.exportStartTimeEpochMillis - VERSION_OVERLAP_TIME_FOR_EXPORT.toMillis()) * 1000L;
                        Event event = this.recordConverter.convert(record, this.exportStartTimeEpochMillis, eventVersionNumber, primaryKeyBsonType);
                        if (this.sourceConfig.getIdKey() != null && !this.sourceConfig.getIdKey().isBlank()) {
                            event.put(this.sourceConfig.getIdKey(), event.get("_id", Object.class));
                        }
                        event.delete("_id");
                        records.add(event);
                        if ((recordCount - this.startLine) % 100 == 0) {
                            LOG.debug("Write to buffer for line " + (recordCount - 100) + " to " + recordCount);
                            this.recordBufferWriter.writeToBuffer(this.acknowledgementSet, records);
                            this.successItemsCounter.increment((double)records.size());
                            this.bytesProcessedSummary.record((double)recordBytes.stream().mapToLong(Long::longValue).sum());
                            records.clear();
                            recordBytes.clear();
                            lastRecordNumberProcessed = recordCount;
                        }
                        if (System.currentTimeMillis() - lastCheckpointTime > 120000L) {
                            LOG.debug("Perform regular checkpointing for Data Query Loader");
                            this.partitionCheckpoint.checkpoint(lastRecordNumberProcessed);
                            lastCheckpointTime = System.currentTimeMillis();
                        }
                        ++successRecords;
                    }
                    catch (Exception e) {
                        LOG.error("Failed to add record to buffer with error.", (Throwable)e);
                        this.failureItemsCounter.increment((double)records.size());
                        ++failedRecords;
                    }
                }
                if (!records.isEmpty()) {
                    this.recordBufferWriter.writeToBuffer(this.acknowledgementSet, records);
                    this.partitionCheckpoint.checkpoint(recordCount);
                    this.successItemsCounter.increment((double)records.size());
                    this.bytesProcessedSummary.record((double)recordBytes.stream().mapToLong(Long::longValue).sum());
                }
                records.clear();
                recordBytes.clear();
                LOG.info("Completed writing query partition: {} to buffer", (Object)query);
                if (this.acknowledgementSet != null) {
                    this.partitionCheckpoint.updateDatafileForAcknowledgmentWait(this.sourceConfig.getPartitionAcknowledgmentTimeout());
                    this.acknowledgementSet.complete();
                }
            }
            catch (Exception e) {
                LOG.error("Exception connecting to cluster and loading partition {}.", (Object)query, (Object)e);
                throw new RuntimeException(e);
            }
            finally {
                this.partitionCheckpoint.checkpoint(recordCount);
            }
            LOG.info("Records processed: {}, recordCount: {}", (Object)totalRecords, (Object)recordCount);
        }
        catch (MongoClientException | IllegalArgumentException e) {
            this.documentDBAggregateMetrics.getExport4xxErrors().increment();
            LOG.error("Client side exception while connecting to cluster and loading partition.", e);
            throw new RuntimeException(e);
        }
        catch (Exception e) {
            this.documentDBAggregateMetrics.getExport5xxErrors().increment();
            LOG.error("Server side exception while connecting to cluster and loading partition.", (Throwable)e);
            throw new RuntimeException(e);
        }
    }

    public static void stopAll() {
        shouldStop = true;
    }
}

