/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.dataprepper.core.pipeline;

import com.google.common.base.Preconditions;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;
import javax.annotation.Nonnull;
import org.opensearch.dataprepper.DataPrepperShutdownOptions;
import org.opensearch.dataprepper.core.acknowledgements.InactiveAcknowledgementSetManager;
import org.opensearch.dataprepper.core.parser.DataFlowComponent;
import org.opensearch.dataprepper.core.pipeline.PipelineObserver;
import org.opensearch.dataprepper.core.pipeline.PipelineShutdown;
import org.opensearch.dataprepper.core.pipeline.ProcessWorker;
import org.opensearch.dataprepper.core.pipeline.ProcessorProvider;
import org.opensearch.dataprepper.core.pipeline.ProcessorRegistry;
import org.opensearch.dataprepper.core.pipeline.common.PipelineThreadFactory;
import org.opensearch.dataprepper.core.pipeline.common.PipelineThreadPoolExecutor;
import org.opensearch.dataprepper.core.pipeline.router.Router;
import org.opensearch.dataprepper.core.pipeline.router.RouterCopyRecordStrategy;
import org.opensearch.dataprepper.core.sourcecoordination.SourceCoordinatorFactory;
import org.opensearch.dataprepper.model.acknowledgements.AcknowledgementSetManager;
import org.opensearch.dataprepper.model.buffer.Buffer;
import org.opensearch.dataprepper.model.event.EventFactory;
import org.opensearch.dataprepper.model.processor.Processor;
import org.opensearch.dataprepper.model.record.Record;
import org.opensearch.dataprepper.model.sink.Sink;
import org.opensearch.dataprepper.model.source.Source;
import org.opensearch.dataprepper.model.source.coordinator.SourceCoordinator;
import org.opensearch.dataprepper.model.source.coordinator.UsesSourceCoordination;
import org.opensearch.dataprepper.model.source.coordinator.enhanced.EnhancedSourceCoordinator;
import org.opensearch.dataprepper.model.source.coordinator.enhanced.UsesEnhancedSourceCoordination;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Pipeline {
    private static final Logger LOG = LoggerFactory.getLogger(Pipeline.class);
    private static final int SINK_LOGGING_FREQUENCY = (int)Duration.ofSeconds(60L).toMillis();
    private final ProcessorRegistry singleThreadUnsafeProcessorRegistry;
    private final PipelineShutdown pipelineShutdown;
    private final String name;
    private final Source source;
    private final Buffer buffer;
    private final List<List<Processor>> processorSets;
    private final List<DataFlowComponent<Sink>> sinks;
    private final Router router;
    private final SourceCoordinatorFactory sourceCoordinatorFactory;
    private final int processorThreads;
    private final int readBatchTimeoutInMillis;
    private final Duration processorShutdownTimeout;
    private final Duration sinkShutdownTimeout;
    private final Duration peerForwarderDrainTimeout;
    private final ExecutorService processorExecutorService;
    private final ExecutorService sinkExecutorService;
    private final EventFactory eventFactory;
    private final AcknowledgementSetManager acknowledgementSetManager;
    private final List<PipelineObserver> observers = Collections.synchronizedList(new LinkedList());

    public Pipeline(@Nonnull String name, @Nonnull Source source, @Nonnull Buffer buffer, @Nonnull List<List<Processor>> processorSets, @Nonnull List<DataFlowComponent<Sink>> sinks, @Nonnull Router router, @Nonnull EventFactory eventFactory, @Nonnull AcknowledgementSetManager acknowledgementSetManager, SourceCoordinatorFactory sourceCoordinatorFactory, int processorThreads, int readBatchTimeoutInMillis, Duration processorShutdownTimeout, Duration sinkShutdownTimeout, Duration peerForwarderDrainTimeout) {
        Preconditions.checkArgument((boolean)processorSets.stream().allMatch(processorSet -> Objects.nonNull(processorSet) && (processorSet.size() == 1 || processorSet.size() == processorThreads)));
        this.name = name;
        this.source = source;
        this.buffer = buffer;
        this.processorSets = processorSets;
        this.sinks = sinks;
        this.router = router;
        this.sourceCoordinatorFactory = sourceCoordinatorFactory;
        this.processorThreads = processorThreads;
        this.eventFactory = eventFactory;
        this.acknowledgementSetManager = acknowledgementSetManager;
        this.readBatchTimeoutInMillis = readBatchTimeoutInMillis;
        this.processorShutdownTimeout = processorShutdownTimeout;
        this.sinkShutdownTimeout = sinkShutdownTimeout;
        this.peerForwarderDrainTimeout = peerForwarderDrainTimeout;
        this.processorExecutorService = PipelineThreadPoolExecutor.newFixedThreadPool(processorThreads, new PipelineThreadFactory(String.format("%s-processor-worker", name)), this);
        this.sinkExecutorService = PipelineThreadPoolExecutor.newFixedThreadPool(processorThreads, new PipelineThreadFactory(String.format("%s-sink-worker", name)), this);
        this.pipelineShutdown = new PipelineShutdown(name, buffer);
        this.singleThreadUnsafeProcessorRegistry = new ProcessorRegistry(List.of());
    }

    public String getName() {
        return this.name;
    }

    public Source getSource() {
        return this.source;
    }

    public Buffer getBuffer() {
        return this.buffer;
    }

    public Collection<Sink> getSinks() {
        return this.sinks.stream().map(DataFlowComponent::getComponent).collect(Collectors.toList());
    }

    public boolean isStopRequested() {
        return this.pipelineShutdown.isStopRequested();
    }

    public boolean isForceStopReadingBuffers() {
        return this.pipelineShutdown.isForceStopReadingBuffers();
    }

    public Duration getPeerForwarderDrainTimeout() {
        return this.peerForwarderDrainTimeout;
    }

    List<List<Processor>> getProcessorSets() {
        return this.processorSets;
    }

    public ProcessorProvider getSingleThreadUnsafeProcessorProvider() {
        return this.singleThreadUnsafeProcessorRegistry;
    }

    public void swapProcessors(List<Processor> newProcessors) {
        this.singleThreadUnsafeProcessorRegistry.swapProcessors(newProcessors);
    }

    public int getReadBatchTimeoutInMillis() {
        return this.readBatchTimeoutInMillis;
    }

    public boolean isReady() {
        for (Sink sink : this.getSinks()) {
            if (sink.isReady()) continue;
            LOG.info("Pipeline [{}] - sink is not ready for execution, retrying", (Object)this.name);
            sink.initialize();
            if (sink.isReady()) continue;
            return false;
        }
        return true;
    }

    private synchronized void startSourceAndProcessors() {
        if (this.isStopRequested()) {
            return;
        }
        LOG.info("Pipeline [{}] Sink is ready, starting source...", (Object)this.name);
        this.source.start(this.buffer);
        LOG.info("Pipeline [{}] - Submitting request to initiate the pipeline processing", (Object)this.name);
        int i = 0;
        while (i < this.processorThreads) {
            int finalI = i++;
            List<Processor> processors = this.processorSets.stream().map(processorSet -> {
                if (processorSet.size() == 1) {
                    return (Processor)processorSet.get(0);
                }
                return (Processor)processorSet.get(finalI);
            }).collect(Collectors.toList());
            ProcessorRegistry workerSpecificProcessorRegistry = new ProcessorRegistry(processors);
            this.processorExecutorService.submit(new ProcessWorker(this.buffer, this, workerSpecificProcessorRegistry));
            this.singleThreadUnsafeProcessorRegistry.swapProcessors(processors);
        }
    }

    public void execute() {
        LOG.info("Pipeline [{}] - Initiating pipeline execution", (Object)this.name);
        try {
            if (this.source instanceof UsesSourceCoordination) {
                Class partionProgressModelClass = ((UsesSourceCoordination)this.source).getPartitionProgressStateClass();
                SourceCoordinator sourceCoordinator = this.sourceCoordinatorFactory.provideSourceCoordinator(partionProgressModelClass, this.name);
                ((UsesSourceCoordination)this.source).setSourceCoordinator(sourceCoordinator);
            } else if (this.source instanceof UsesEnhancedSourceCoordination) {
                Function partitionFactory = ((UsesEnhancedSourceCoordination)this.source).getPartitionFactory();
                EnhancedSourceCoordinator enhancedSourceCoordinator = this.sourceCoordinatorFactory.provideEnhancedSourceCoordinator(partitionFactory, this.name);
                ((UsesEnhancedSourceCoordination)this.source).setEnhancedSourceCoordinator(enhancedSourceCoordinator);
            }
            this.sinkExecutorService.submit(() -> {
                long retryCount = 0L;
                long sleepIfNotReadyTime = 200L;
                while (!this.isReady() && !this.isStopRequested()) {
                    if (retryCount++ % ((long)SINK_LOGGING_FREQUENCY / 200L) == 0L) {
                        LOG.info("Pipeline [{}] Waiting for Sink to be ready", (Object)this.name);
                    }
                    try {
                        Thread.sleep(200L);
                    }
                    catch (Exception exception) {}
                }
                this.startSourceAndProcessors();
            }, null);
        }
        catch (Exception ex) {
            LOG.error("Pipeline [{}] encountered exception while starting the source, skipping execution", (Object)this.name, (Object)ex);
        }
    }

    public synchronized void shutdown() {
        this.shutdown(DataPrepperShutdownOptions.defaultOptions());
    }

    public synchronized void shutdown(DataPrepperShutdownOptions dataPrepperShutdownOptions) {
        LOG.info("Pipeline [{}] - Received shutdown signal with buffer drain timeout {}, processor shutdown timeout {}, and sink shutdown timeout {}. Initiating the shutdown process", new Object[]{this.name, this.buffer.getDrainTimeout(), this.processorShutdownTimeout, this.sinkShutdownTimeout});
        try {
            this.source.stop();
        }
        catch (Exception ex) {
            LOG.error("Pipeline [{}] - Encountered exception while stopping the source, proceeding with termination of process workers", (Object)this.name, (Object)ex);
        }
        this.pipelineShutdown.shutdown(dataPrepperShutdownOptions);
        this.shutdownExecutorService(this.processorExecutorService, this.pipelineShutdown.getBufferDrainTimeout().plus(this.processorShutdownTimeout), "processor");
        this.processorSets.forEach(processorSet -> processorSet.forEach(Processor::shutdown));
        this.buffer.shutdown();
        this.sinks.stream().map(DataFlowComponent::getComponent).forEach(Sink::shutdown);
        this.shutdownExecutorService(this.sinkExecutorService, this.sinkShutdownTimeout, "sink");
        LOG.info("Pipeline [{}] - Pipeline fully shutdown.", (Object)this.name);
        this.observers.forEach(observer -> observer.shutdown(this));
    }

    public void addShutdownObserver(PipelineObserver pipelineObserver) {
        this.observers.add(pipelineObserver);
    }

    public void removeShutdownObserver(PipelineObserver pipelineObserver) {
        this.observers.remove(pipelineObserver);
    }

    private void shutdownExecutorService(ExecutorService executorService, Duration timeoutForTermination, String workerName) {
        LOG.info("Pipeline [{}] - Shutting down {} process workers.", (Object)this.name, (Object)workerName);
        executorService.shutdown();
        try {
            if (!executorService.awaitTermination(timeoutForTermination.toMillis(), TimeUnit.MILLISECONDS)) {
                LOG.warn("Pipeline [{}] - Workers did not terminate in {}, forcing termination of {} workers.", new Object[]{this.name, timeoutForTermination, workerName});
                executorService.shutdownNow();
            }
        }
        catch (InterruptedException ex) {
            LOG.info("Pipeline [{}] - Encountered interruption terminating the pipeline execution, Attempting to force the termination of {} workers.", (Object)this.name, (Object)workerName);
            executorService.shutdownNow();
        }
    }

    public List<Future<Void>> publishToSinks(Collection<Record> records) {
        int sinksSize = this.sinks.size();
        ArrayList<Future<Void>> sinkFutures = new ArrayList<Future<Void>>(sinksSize);
        RouterCopyRecordStrategy getRecordStrategy = new RouterCopyRecordStrategy(this.eventFactory, (AcknowledgementSetManager)(this.source.areAcknowledgementsEnabled() || this.buffer.areAcknowledgementsEnabled() ? this.acknowledgementSetManager : InactiveAcknowledgementSetManager.getInstance()), this.sinks);
        this.router.route(records, this.sinks, getRecordStrategy, (sink, events) -> sinkFutures.add(this.sinkExecutorService.submit(() -> {
            sink.updateLatencyMetrics(events);
            sink.output(events);
        }, null)));
        return sinkFutures;
    }

    public boolean areAcknowledgementsEnabled() {
        return this.source.areAcknowledgementsEnabled() || this.buffer.areAcknowledgementsEnabled();
    }
}

