/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.dataprepper.core.peerforwarder;

import com.google.common.base.Stopwatch;
import com.google.common.util.concurrent.AtomicDouble;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Map;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.opensearch.dataprepper.model.CheckpointState;
import org.opensearch.dataprepper.model.buffer.AbstractBuffer;
import org.opensearch.dataprepper.model.buffer.SizeOverflowException;
import org.opensearch.dataprepper.model.record.Record;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PeerForwarderReceiveBuffer<T extends Record<?>>
extends AbstractBuffer<T> {
    private static final Logger LOG = LoggerFactory.getLogger(PeerForwarderReceiveBuffer.class);
    private static final String CORE_PEER_FORWARDER_COMPONENT = "core.peerForwarder";
    private static final String BUFFER_ID_FORMAT = "%s.%s";
    private static final String BUFFER_USAGE_METRIC = "bufferUsage";
    private final int bufferSize;
    private final int batchSize;
    private final Semaphore capacitySemaphore;
    private final LinkedBlockingQueue<T> blockingQueue;
    private int recordsInFlight = 0;
    private final AtomicDouble bufferUsage;

    public PeerForwarderReceiveBuffer(int bufferSize, int batchSize, String pipelineName, String pluginId) {
        super(String.format(BUFFER_ID_FORMAT, pipelineName, pluginId), CORE_PEER_FORWARDER_COMPONENT);
        this.bufferSize = bufferSize;
        this.batchSize = batchSize;
        this.blockingQueue = new LinkedBlockingQueue(bufferSize);
        this.capacitySemaphore = new Semaphore(bufferSize);
        this.bufferUsage = (AtomicDouble)this.pluginMetrics.gauge(BUFFER_USAGE_METRIC, (Number)new AtomicDouble());
    }

    public void doWrite(T record, int timeoutInMillis) throws TimeoutException {
        try {
            boolean permitAcquired = this.capacitySemaphore.tryAcquire(timeoutInMillis, TimeUnit.MILLISECONDS);
            if (!permitAcquired) {
                throw new TimeoutException("Peer forwarder buffer is full, timed out waiting for a slot");
            }
            this.blockingQueue.offer(record);
        }
        catch (InterruptedException ex) {
            LOG.error("Peer forwarder buffer is full, interrupted while waiting to write the record", (Throwable)ex);
            throw new TimeoutException("Peer forwarder buffer is full, timed out waiting for a slot");
        }
    }

    public void doWriteAll(Collection<T> records, int timeoutInMillis) throws Exception {
        int size = records.size();
        if (size > this.bufferSize) {
            throw new SizeOverflowException(String.format("Peer forwarder buffer capacity too small for the number of records: %d", size));
        }
        try {
            boolean permitAcquired = this.capacitySemaphore.tryAcquire(size, timeoutInMillis, TimeUnit.MILLISECONDS);
            if (!permitAcquired) {
                throw new TimeoutException(String.format("Peer forwarder buffer does not have enough capacity left for the number of records: %d, timed out waiting for slots.", size));
            }
            this.blockingQueue.addAll(records);
        }
        catch (InterruptedException ex) {
            LOG.error("Peer forwarder buffer does not have enough capacity left for the number of records: {}, interrupted while waiting to write the records", (Object)size, (Object)ex);
            throw new TimeoutException(String.format("Peer forwarder buffer does not have enough capacity left for the number of records: %d, timed out waiting for slots.", size));
        }
    }

    public Map.Entry<Collection<T>, CheckpointState> doRead(int timeoutInMillis) {
        ArrayList<T> records = new ArrayList<T>(this.batchSize);
        int recordsRead = 0;
        if (timeoutInMillis == 0) {
            T record = this.pollForBufferEntry(5, TimeUnit.MILLISECONDS);
            if (record != null) {
                records.add(record);
                ++recordsRead;
            }
            recordsRead += this.blockingQueue.drainTo(records, this.batchSize - 1);
        } else {
            Stopwatch stopwatch = Stopwatch.createStarted();
            while (stopwatch.elapsed(TimeUnit.MILLISECONDS) < (long)timeoutInMillis && records.size() < this.batchSize) {
                T record = this.pollForBufferEntry(timeoutInMillis, TimeUnit.MILLISECONDS);
                if (record != null) {
                    records.add(record);
                    ++recordsRead;
                }
                if (recordsRead >= this.batchSize) continue;
                recordsRead += this.blockingQueue.drainTo(records, this.batchSize - recordsRead);
            }
        }
        this.updateLatency(records);
        CheckpointState checkpointState = new CheckpointState(recordsRead);
        this.recordsInFlight += checkpointState.getNumRecordsToBeChecked();
        return new AbstractMap.SimpleEntry<Collection<T>, CheckpointState>(records, checkpointState);
    }

    private T pollForBufferEntry(int timeoutValue, TimeUnit timeoutUnit) {
        try {
            return (T)((Record)this.blockingQueue.poll(timeoutValue, timeoutUnit));
        }
        catch (InterruptedException e) {
            LOG.info("Peer forwarder buffer - Interrupt received while reading from buffer");
            throw new RuntimeException(e);
        }
    }

    public void postProcess(Long recordsInBuffer) {
        Double nonNegativeTotalRecords = recordsInBuffer.doubleValue() < 0.0 ? 0.0 : recordsInBuffer.doubleValue();
        Double boundedTotalRecords = nonNegativeTotalRecords > (double)this.bufferSize ? (double)this.bufferSize : nonNegativeTotalRecords;
        Double usage = boundedTotalRecords / (double)this.bufferSize * 100.0;
        this.bufferUsage.set(usage.doubleValue());
    }

    public void doCheckpoint(CheckpointState checkpointState) {
        int numCheckedRecords = checkpointState.getNumRecordsToBeChecked();
        this.capacitySemaphore.release(numCheckedRecords);
        this.recordsInFlight -= checkpointState.getNumRecordsToBeChecked();
    }

    public boolean isEmpty() {
        return this.blockingQueue.isEmpty() && this.recordsInFlight == 0;
    }
}

