/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.dataprepper.plugins.geoip.extension;

import com.linecorp.armeria.client.retry.Backoff;
import java.io.File;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.opensearch.dataprepper.plugins.geoip.exception.DownloadFailedException;
import org.opensearch.dataprepper.plugins.geoip.exception.NoValidDatabaseFoundException;
import org.opensearch.dataprepper.plugins.geoip.extension.AutoCountingDatabaseReader;
import org.opensearch.dataprepper.plugins.geoip.extension.DatabaseSourceIdentification;
import org.opensearch.dataprepper.plugins.geoip.extension.GeoIP2DatabaseReader;
import org.opensearch.dataprepper.plugins.geoip.extension.GeoLite2DatabaseReader;
import org.opensearch.dataprepper.plugins.geoip.extension.LicenseTypeCheck;
import org.opensearch.dataprepper.plugins.geoip.extension.MaxMindConfig;
import org.opensearch.dataprepper.plugins.geoip.extension.MaxMindDatabaseConfig;
import org.opensearch.dataprepper.plugins.geoip.extension.api.GeoIPDatabaseReader;
import org.opensearch.dataprepper.plugins.geoip.extension.databasedownload.DBSourceOptions;
import org.opensearch.dataprepper.plugins.geoip.extension.databasedownload.DatabaseReaderBuilder;
import org.opensearch.dataprepper.plugins.geoip.extension.databasedownload.GeoIPFileManager;
import org.opensearch.dataprepper.plugins.geoip.extension.databasedownload.HttpDBDownloadService;
import org.opensearch.dataprepper.plugins.geoip.extension.databasedownload.LicenseTypeOptions;
import org.opensearch.dataprepper.plugins.geoip.extension.databasedownload.LocalDBDownloadService;
import org.opensearch.dataprepper.plugins.geoip.extension.databasedownload.ManifestDownloadService;
import org.opensearch.dataprepper.plugins.geoip.extension.databasedownload.S3DBService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class GeoIPDatabaseManager {
    private static final Logger LOG = LoggerFactory.getLogger(GeoIPDatabaseManager.class);
    public static final String BLUE_DATABASE_DIR = "blue_database";
    public static final String GREEN_DATABASE_DIR = "green_database";
    private static final long INITIAL_DELAY = Duration.ofMinutes(1L).toMillis();
    private static final long MAXIMUM_DELAY = Duration.ofHours(1L).toMillis();
    private static final double JITTER_RATE = 0.15;
    private final MaxMindConfig maxMindConfig;
    private final LicenseTypeCheck licenseTypeCheck;
    private final DatabaseReaderBuilder databaseReaderBuilder;
    private final MaxMindDatabaseConfig maxMindDatabaseConfig;
    private final ReentrantReadWriteLock.WriteLock writeLock;
    private final int cacheSize;
    private final GeoIPFileManager geoIPFileManager;
    private final AtomicInteger failedAttemptCount;
    private final Backoff backoff;
    private final DBSourceOptions dbSourceOptions;
    private String currentDatabaseDir;
    private GeoIPDatabaseReader geoIPDatabaseReader;
    private boolean databaseDirToggle;
    private Instant nextUpdateAt;

    public GeoIPDatabaseManager(MaxMindConfig maxMindConfig, LicenseTypeCheck licenseTypeCheck, DatabaseReaderBuilder databaseReaderBuilder, GeoIPFileManager geoIPFileManager, ReentrantReadWriteLock.WriteLock writeLock) {
        this.maxMindConfig = maxMindConfig;
        this.licenseTypeCheck = licenseTypeCheck;
        this.databaseReaderBuilder = databaseReaderBuilder;
        this.geoIPFileManager = geoIPFileManager;
        this.maxMindDatabaseConfig = maxMindConfig.getMaxMindDatabaseConfig();
        this.writeLock = writeLock;
        this.cacheSize = maxMindConfig.getCacheSize();
        this.failedAttemptCount = new AtomicInteger(0);
        this.backoff = Backoff.exponential((long)INITIAL_DELAY, (long)MAXIMUM_DELAY).withJitter(0.15).withMaxAttempts(Integer.MAX_VALUE);
        this.dbSourceOptions = DatabaseSourceIdentification.getDatabasePathType(new ArrayList<String>(this.maxMindDatabaseConfig.getDatabasePaths().values()));
    }

    public void initiateDatabaseDownload() {
        try {
            this.downloadDatabases();
            this.geoIPDatabaseReader = this.createReader();
            this.nextUpdateAt = Instant.now().plus(this.maxMindConfig.getDatabaseRefreshInterval());
            this.failedAttemptCount.set(0);
        }
        catch (Exception e) {
            Duration delay = Duration.ofMillis(this.applyBackoff());
            this.nextUpdateAt = Instant.now().plus(delay);
            throw new DownloadFailedException(e.getMessage());
        }
    }

    public void updateDatabaseReader() {
        try {
            this.downloadDatabases();
            this.switchDatabase();
            LOG.info("Updated geoip database readers");
            this.failedAttemptCount.set(0);
        }
        catch (Exception e) {
            LOG.error("Failed to download database and create database readers, will try to use old databases if they exist. {}", (Object)e.getMessage());
            Duration delay = Duration.ofMillis(this.applyBackoff());
            this.nextUpdateAt = Instant.now().plus(delay);
            File file = new File(this.currentDatabaseDir);
            this.geoIPFileManager.deleteDirectory(file);
            this.switchDirectory();
        }
    }

    private void switchDatabase() {
        this.writeLock.lock();
        try {
            GeoIPDatabaseReader newGeoipDatabaseReader = this.createReader();
            GeoIPDatabaseReader oldGeoipDatabaseReader = this.geoIPDatabaseReader;
            this.geoIPDatabaseReader = newGeoipDatabaseReader;
            if (oldGeoipDatabaseReader != null) {
                oldGeoipDatabaseReader.close();
            }
        }
        catch (Exception e) {
            LOG.error("Failed to close geoip database readers due to: {}", (Object)e.getMessage());
        }
        finally {
            this.writeLock.unlock();
        }
    }

    private void downloadDatabases() throws Exception {
        this.switchDirectory();
        String destinationPath = this.maxMindConfig.getDatabaseDestination() + File.separator + this.currentDatabaseDir;
        this.geoIPFileManager.createDirectoryIfNotExist(destinationPath);
        LOG.info("Downloading GeoIP database to {}", (Object)destinationPath);
        switch (this.dbSourceOptions) {
            case HTTP_MANIFEST: {
                ManifestDownloadService dbSource = new ManifestDownloadService(destinationPath, this.maxMindDatabaseConfig);
                dbSource.initiateDownload();
                break;
            }
            case URL: {
                HttpDBDownloadService dbSource = new HttpDBDownloadService(destinationPath, this.geoIPFileManager, this.maxMindDatabaseConfig);
                dbSource.initiateDownload();
                break;
            }
            case S3: {
                S3DBService dbSource = new S3DBService(this.maxMindConfig.getAwsAuthenticationOptionsConfig(), destinationPath, this.maxMindDatabaseConfig);
                dbSource.initiateDownload();
                break;
            }
            case PATH: {
                LocalDBDownloadService dbSource = new LocalDBDownloadService(destinationPath, this.maxMindDatabaseConfig);
                dbSource.initiateDownload();
            }
        }
    }

    private GeoIPDatabaseReader createReader() {
        AutoCountingDatabaseReader newGeoIPDatabaseReader;
        String finalPath = this.maxMindConfig.getDatabaseDestination() + File.separator + this.currentDatabaseDir;
        LicenseTypeOptions licenseType = this.licenseTypeCheck.isGeoLite2OrEnterpriseLicense(finalPath);
        if (licenseType == null) {
            throw new NoValidDatabaseFoundException("At least one valid database is required.");
        }
        if (licenseType.equals((Object)LicenseTypeOptions.FREE)) {
            newGeoIPDatabaseReader = new AutoCountingDatabaseReader(new GeoLite2DatabaseReader(this.databaseReaderBuilder, this.geoIPFileManager, finalPath, this.cacheSize));
        } else if (licenseType.equals((Object)LicenseTypeOptions.ENTERPRISE)) {
            newGeoIPDatabaseReader = new AutoCountingDatabaseReader(new GeoIP2DatabaseReader(this.databaseReaderBuilder, this.geoIPFileManager, finalPath, this.cacheSize));
        } else {
            throw new NoValidDatabaseFoundException("No valid database found to initialize database readers.");
        }
        return newGeoIPDatabaseReader;
    }

    private void switchDirectory() {
        this.databaseDirToggle = !this.databaseDirToggle;
        this.currentDatabaseDir = this.databaseDirToggle ? BLUE_DATABASE_DIR : GREEN_DATABASE_DIR;
    }

    private long applyBackoff() {
        long delayMillis = this.backoff.nextDelayMillis(this.failedAttemptCount.incrementAndGet());
        if (delayMillis < 0L) {
            LOG.info("Retries exhausted to download database. Will retry based on refresh interval");
        }
        Duration delayDuration = Duration.ofMillis(delayMillis);
        LOG.info("Failed to download databases, will retry after {} seconds", (Object)delayDuration.getSeconds());
        return delayMillis;
    }

    public GeoIPDatabaseReader getGeoIPDatabaseReader() {
        return this.geoIPDatabaseReader;
    }

    public Instant getNextUpdateAt() {
        return this.nextUpdateAt;
    }

    public void setNextUpdateAt(Instant nextUpdateAt) {
        this.nextUpdateAt = nextUpdateAt;
    }

    public void deleteDatabasesOnShutdown() {
        this.geoIPFileManager.deleteDirectory(new File(this.maxMindConfig.getDatabaseDestination() + File.separator + BLUE_DATABASE_DIR));
        this.geoIPFileManager.deleteDirectory(new File(this.maxMindConfig.getDatabaseDestination() + File.separator + GREEN_DATABASE_DIR));
    }

    public void deleteDirectory(File file) {
        if (file.exists()) {
            for (File subFile : file.listFiles()) {
                if (subFile.isDirectory()) {
                    this.deleteDirectory(subFile);
                }
                subFile.delete();
            }
            file.delete();
        }
    }
}

