/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.dataprepper.plugins.sink.otlp.buffer;

import com.google.common.annotations.VisibleForTesting;
import io.opentelemetry.proto.trace.v1.ResourceSpans;
import java.util.ArrayList;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import javax.annotation.Nonnull;
import org.opensearch.dataprepper.aws.api.AwsCredentialsSupplier;
import org.opensearch.dataprepper.model.event.EventHandle;
import org.opensearch.dataprepper.model.record.Record;
import org.opensearch.dataprepper.model.trace.Span;
import org.opensearch.dataprepper.plugins.otel.codec.OTelProtoStandardCodec;
import org.opensearch.dataprepper.plugins.sink.otlp.configuration.OtlpSinkConfig;
import org.opensearch.dataprepper.plugins.sink.otlp.http.OtlpHttpSender;
import org.opensearch.dataprepper.plugins.sink.otlp.metrics.OtlpSinkMetrics;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.utils.Pair;

public class OtlpSinkBuffer {
    private static final Logger LOG = LoggerFactory.getLogger(OtlpSinkBuffer.class);
    private static final int SAFETY_FACTOR = 10;
    private static final int MIN_QUEUE_CAPACITY = 2000;
    private final BlockingQueue<Record<Span>> queue;
    private final OTelProtoStandardCodec.OTelProtoEncoder encoder;
    private final OtlpHttpSender sender;
    private final OtlpSinkMetrics sinkMetrics;
    private final int maxEvents;
    private final long maxBatchBytes;
    private final long flushTimeoutMillis;
    private final ExecutorService executor;
    private volatile boolean running = true;

    public OtlpSinkBuffer(@Nonnull AwsCredentialsSupplier awsCredentialsSupplier, @Nonnull OtlpSinkConfig config, @Nonnull OtlpSinkMetrics sinkMetrics) {
        this(config, sinkMetrics, new OTelProtoStandardCodec.OTelProtoEncoder(), new OtlpHttpSender(awsCredentialsSupplier, config, sinkMetrics));
    }

    @VisibleForTesting
    OtlpSinkBuffer(@Nonnull OtlpSinkConfig config, @Nonnull OtlpSinkMetrics sinkMetrics, @Nonnull OTelProtoStandardCodec.OTelProtoEncoder encoder, @Nonnull OtlpHttpSender sender) {
        this.sinkMetrics = sinkMetrics;
        this.encoder = encoder;
        this.sender = sender;
        this.maxEvents = config.getMaxEvents();
        this.maxBatchBytes = config.getMaxBatchSize();
        this.flushTimeoutMillis = config.getFlushTimeoutMillis();
        this.queue = new LinkedBlockingQueue<Record<Span>>(this.getQueueCapacity());
        sinkMetrics.registerQueueGauges(this.queue);
        this.executor = Executors.newSingleThreadExecutor(r -> {
            Thread t = new Thread(() -> {
                try {
                    r.run();
                }
                catch (Throwable t1) {
                    LOG.error("Worker thread crashed unexpectedly", t1);
                    sinkMetrics.incrementErrorsCount();
                    this.restartWorker();
                }
            }, "otlp-sink-buffer-thread");
            t.setDaemon(false);
            return t;
        });
    }

    private int getQueueCapacity() {
        return Math.max(this.maxEvents * 10, 2000);
    }

    public void start() {
        this.running = true;
        this.executor.execute(this::run);
    }

    public void stop() {
        this.running = false;
        this.executor.shutdownNow();
    }

    @VisibleForTesting
    void restartWorker() {
        if (this.running && !this.executor.isShutdown()) {
            LOG.info("Restarting OTLP sink buffer worker thread");
            this.executor.execute(this::run);
        }
    }

    public void add(Record<Span> record) {
        try {
            this.queue.put(record);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            LOG.error("Interrupted while enqueuing span", (Throwable)e);
            this.sinkMetrics.incrementFailedSpansCount(1L);
            this.sinkMetrics.incrementErrorsCount();
        }
    }

    private void run() {
        ArrayList<Pair<ResourceSpans, EventHandle>> batch = new ArrayList<Pair<ResourceSpans, EventHandle>>();
        long batchSize = 0L;
        long lastFlush = System.currentTimeMillis();
        while (true) {
            try {
                do {
                    boolean flushByTime;
                    long now = System.currentTimeMillis();
                    Record<Span> record = this.queue.poll(100L, TimeUnit.MILLISECONDS);
                    if (record != null) {
                        try {
                            ResourceSpans resourceSpans = this.encoder.convertToResourceSpans((Span)record.getData());
                            EventHandle eventHandle = ((Span)record.getData()).getEventHandle();
                            batch.add(Pair.of((Object)resourceSpans, (Object)eventHandle));
                            batchSize += (long)resourceSpans.getSerializedSize();
                        }
                        catch (Exception e) {
                            LOG.error("Failed to encode span, skipping", (Throwable)e);
                            this.sinkMetrics.incrementFailedSpansCount(1L);
                            this.sinkMetrics.incrementErrorsCount();
                        }
                    }
                    boolean flushBySize = this.maxEvents > 0 && batch.size() >= this.maxEvents || batchSize >= this.maxBatchBytes;
                    boolean bl = flushByTime = !batch.isEmpty() && now - lastFlush >= this.flushTimeoutMillis;
                    if (!flushBySize && !flushByTime) continue;
                    this.sender.send(batch);
                    batch.clear();
                    batchSize = 0L;
                    lastFlush = now;
                } while (this.running || !this.queue.isEmpty());
            }
            catch (InterruptedException e) {
                if (!this.running) continue;
                LOG.debug("Worker interrupted while polling, continuing...");
                this.sinkMetrics.incrementErrorsCount();
                continue;
            }
            break;
        }
        if (!batch.isEmpty()) {
            this.sender.send(batch);
            batch.clear();
        }
    }

    public boolean isRunning() {
        return this.running;
    }
}

