/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.dataprepper.plugins.sink.opensearch.index;

import com.fasterxml.jackson.databind.node.ObjectNode;
import com.google.common.annotations.VisibleForTesting;
import io.micrometer.core.instrument.Counter;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.opensearch.client.opensearch.OpenSearchClient;
import org.opensearch.client.opensearch._types.FieldValue;
import org.opensearch.client.opensearch._types.query_dsl.Query;
import org.opensearch.client.opensearch._types.query_dsl.TermsQuery;
import org.opensearch.client.opensearch._types.query_dsl.TermsQueryField;
import org.opensearch.client.opensearch.core.MsearchRequest;
import org.opensearch.client.opensearch.core.MsearchResponse;
import org.opensearch.dataprepper.metrics.PluginMetrics;
import org.opensearch.dataprepper.plugins.sink.opensearch.BulkOperationWrapper;
import org.opensearch.dataprepper.plugins.sink.opensearch.index.IndexConfiguration;
import org.opensearch.dataprepper.plugins.sink.opensearch.index.model.QueryManagerBulkOperation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ExistingDocumentQueryManager
implements Runnable {
    private static final Logger LOG = LoggerFactory.getLogger(ExistingDocumentQueryManager.class);
    private static final Duration QUERY_INTERVAL = Duration.ofSeconds(20L);
    static final String EVENTS_DROPPED_AND_RELEASED = "eventsDroppedAndReleasedAfterQuery";
    static final String EVENTS_ADDED_FOR_QUERYING = "eventsAddedForQuerying";
    static final String EVENTS_RETURNED_FOR_INDEXING = "eventsReturnedForIndexingAfterQuery";
    static final String DOCUMENTS_CURRENTLY_BEING_QUERIED = "documentsBeingQueried";
    private final Counter eventsDroppedAndReleasedCounter;
    private final Counter eventsAddedForQuerying;
    private final Counter eventsReturnedForIndexing;
    private final AtomicInteger documentsCurrentlyBeingQueried = new AtomicInteger(0);
    private final AtomicInteger documentsCurrentlyBeingQueriedGauge;
    private final IndexConfiguration indexConfiguration;
    private final Map<String, Map<String, QueryManagerBulkOperation>> bulkOperationsWaitingForQuery;
    private Set<BulkOperationWrapper> bulkOperationsReadyToIngest;
    private final PluginMetrics pluginMetrics;
    private final OpenSearchClient openSearchClient;
    private boolean shouldStop = false;
    private Instant lastQueryTime;
    private final Lock lockReadyToIngest;
    private final Lock lockWaitingForQuery;
    private final String queryTerm;

    public ExistingDocumentQueryManager(IndexConfiguration indexConfiguration, PluginMetrics pluginMetrics, OpenSearchClient openSearchClient) {
        this.indexConfiguration = indexConfiguration;
        this.queryTerm = indexConfiguration.getQueryTerm();
        this.bulkOperationsWaitingForQuery = new ConcurrentHashMap<String, Map<String, QueryManagerBulkOperation>>();
        this.bulkOperationsReadyToIngest = ConcurrentHashMap.newKeySet();
        this.pluginMetrics = pluginMetrics;
        this.openSearchClient = openSearchClient;
        this.eventsAddedForQuerying = pluginMetrics.counter(EVENTS_ADDED_FOR_QUERYING);
        this.eventsDroppedAndReleasedCounter = pluginMetrics.counter(EVENTS_DROPPED_AND_RELEASED);
        this.eventsReturnedForIndexing = pluginMetrics.counter(EVENTS_RETURNED_FOR_INDEXING);
        this.documentsCurrentlyBeingQueriedGauge = (AtomicInteger)pluginMetrics.gauge(DOCUMENTS_CURRENTLY_BEING_QUERIED, (Object)this.documentsCurrentlyBeingQueried, AtomicInteger::get);
        this.lockReadyToIngest = new ReentrantLock();
        this.lockWaitingForQuery = new ReentrantLock();
    }

    @Override
    public void run() {
        while (!Thread.currentThread().isInterrupted() && !this.shouldStop) {
            try {
                this.runQueryLoop();
            }
            catch (Exception e) {
                LOG.error("Exception in primary loop responsible for querying for existing documents, retrying", (Throwable)e);
            }
            finally {
                try {
                    Thread.sleep(QUERY_INTERVAL.toMillis());
                }
                catch (InterruptedException e) {
                    LOG.error("Interrupted, exiting");
                }
            }
        }
    }

    @VisibleForTesting
    void runQueryLoop() {
        if (!this.bulkOperationsWaitingForQuery.isEmpty()) {
            MsearchRequest msearchRequest = this.buildMultiSearchRequest();
            MsearchResponse<ObjectNode> msearchResponse = this.queryForTermValues(msearchRequest);
            this.lastQueryTime = Instant.now();
            this.dropAndReleaseFoundEvents(msearchResponse);
            this.moveBulkRequestsThatHaveReachedQueryDuration();
        }
    }

    public void stop() {
        this.shouldStop = true;
    }

    public void addBulkOperation(BulkOperationWrapper bulkOperationWrapper) {
        this.lockWaitingForQuery.lock();
        String termValue = bulkOperationWrapper.getTermValue();
        try {
            this.bulkOperationsWaitingForQuery.computeIfAbsent(bulkOperationWrapper.getIndex(), k -> new ConcurrentHashMap()).put(termValue, new QueryManagerBulkOperation(bulkOperationWrapper, Instant.now(), termValue));
        }
        finally {
            this.lockWaitingForQuery.unlock();
        }
        this.documentsCurrentlyBeingQueriedGauge.incrementAndGet();
        this.eventsAddedForQuerying.increment();
    }

    public Set<BulkOperationWrapper> getAndClearBulkOperationsReadyToIndex() {
        while (this.documentsCurrentlyBeingQueried.get() > this.indexConfiguration.getQueryAsyncDocumentLimit()) {
            try {
                Thread.sleep(QUERY_INTERVAL.toMillis());
            }
            catch (InterruptedException e) {
                LOG.warn("Interrupted while waiting for documents currently being queried to be under limit {}", (Object)this.indexConfiguration.getQueryAsyncDocumentLimit());
            }
        }
        this.lockReadyToIngest.lock();
        try {
            Set<BulkOperationWrapper> copyOfBulkOperations = this.bulkOperationsReadyToIngest;
            this.bulkOperationsReadyToIngest = ConcurrentHashMap.newKeySet();
            this.eventsReturnedForIndexing.increment((double)copyOfBulkOperations.size());
            Set<BulkOperationWrapper> set = copyOfBulkOperations;
            return set;
        }
        finally {
            this.lockReadyToIngest.unlock();
        }
    }

    private MsearchRequest buildMultiSearchRequest() {
        return MsearchRequest.of(m -> {
            for (Map.Entry<String, Map<String, QueryManagerBulkOperation>> entry : this.bulkOperationsWaitingForQuery.entrySet()) {
                String index = entry.getKey();
                List<FieldValue> values = this.getTermValues(entry.getValue().values());
                m.searches(s -> s.header(h -> h.index(index, new String[0])).body(b -> b.size(Integer.valueOf(values.size())).source(source -> source.filter(f -> f.includes(this.queryTerm, new String[0]))).query(Query.of(q -> q.terms(TermsQuery.of(t -> t.field(this.queryTerm).terms(TermsQueryField.of(tf -> tf.value(values)))))))));
            }
            return m;
        });
    }

    private MsearchResponse<ObjectNode> queryForTermValues(MsearchRequest searchRequest) {
        try {
            return this.openSearchClient.msearch(searchRequest, ObjectNode.class);
        }
        catch (Exception e) {
            LOG.error("Exception while querying for indexes {}", (Object)searchRequest.index());
            throw new RuntimeException(String.format("Exception while querying for indexes %s: %s", searchRequest.index(), e.getMessage()));
        }
    }

    private List<FieldValue> getTermValues(Collection<QueryManagerBulkOperation> queryManagerBulkOperations) {
        ArrayList<FieldValue> termValues = new ArrayList<FieldValue>();
        for (QueryManagerBulkOperation queryManagerBulkOperation : queryManagerBulkOperations) {
            termValues.add(FieldValue.of((String)queryManagerBulkOperation.getTermValue()));
        }
        return termValues;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void moveBulkRequestsThatHaveReachedQueryDuration() {
        for (Map.Entry<String, Map<String, QueryManagerBulkOperation>> operationsForEachIndex : this.bulkOperationsWaitingForQuery.entrySet()) {
            Iterator<Map.Entry<String, QueryManagerBulkOperation>> bulkOperationIterator = operationsForEachIndex.getValue().entrySet().iterator();
            while (bulkOperationIterator.hasNext()) {
                Map.Entry<String, QueryManagerBulkOperation> entry = bulkOperationIterator.next();
                QueryManagerBulkOperation bulkOperation = entry.getValue();
                if (!bulkOperation.getStartTime().plus(this.indexConfiguration.getQueryDuration()).isBefore(this.lastQueryTime)) continue;
                this.lockReadyToIngest.lock();
                try {
                    LOG.debug("Moving bulk operation for index {} and term value {} to be ingested after querying and finding no existing document", (Object)bulkOperation.getBulkOperationWrapper().getIndex(), (Object)bulkOperation.getTermValue());
                    this.bulkOperationsReadyToIngest.add(bulkOperation.getBulkOperationWrapper());
                    bulkOperationIterator.remove();
                    this.documentsCurrentlyBeingQueriedGauge.decrementAndGet();
                }
                finally {
                    this.lockReadyToIngest.unlock();
                }
            }
        }
    }

    private void dropAndReleaseFoundEvents(MsearchResponse<ObjectNode> msearchResponse) {
        msearchResponse.responses().forEach(response -> {
            if (response.isFailure()) {
                LOG.error("Search response failed: {}", (Object)response.failure().error().reason());
            } else {
                response.result().hits().hits().forEach(hit -> {
                    String indexForHit = hit.index();
                    ObjectNode sourceForHit = (ObjectNode)hit.source();
                    String queryTermValue = sourceForHit.findValue(this.queryTerm).textValue();
                    this.lockWaitingForQuery.lock();
                    try {
                        Map<String, QueryManagerBulkOperation> bulkOperationsForIndex = this.bulkOperationsWaitingForQuery.get(indexForHit);
                        QueryManagerBulkOperation bulkOperationToRelease = bulkOperationsForIndex.get(queryTermValue);
                        if (bulkOperationToRelease == null) {
                            LOG.error("bulk operation for term value {} is null", (Object)queryTermValue);
                        } else {
                            LOG.debug("Found document with query term {}, dropping and releasing Event handle", (Object)queryTermValue);
                            bulkOperationToRelease.getBulkOperationWrapper().releaseEventHandle(true);
                            this.eventsDroppedAndReleasedCounter.increment();
                            bulkOperationsForIndex.remove(queryTermValue);
                        }
                    }
                    finally {
                        this.lockWaitingForQuery.unlock();
                    }
                });
            }
        });
    }
}

