/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.dataprepper.plugins.source.dynamodb.stream;

import io.micrometer.core.instrument.Counter;
import java.time.Duration;
import java.time.Instant;
import java.util.List;
import java.util.stream.Collectors;
import org.opensearch.dataprepper.buffer.common.BufferAccumulator;
import org.opensearch.dataprepper.metrics.PluginMetrics;
import org.opensearch.dataprepper.model.acknowledgements.AcknowledgementSet;
import org.opensearch.dataprepper.model.buffer.Buffer;
import org.opensearch.dataprepper.model.event.Event;
import org.opensearch.dataprepper.model.record.Record;
import org.opensearch.dataprepper.plugins.source.dynamodb.configuration.StreamConfig;
import org.opensearch.dataprepper.plugins.source.dynamodb.converter.StreamRecordConverter;
import org.opensearch.dataprepper.plugins.source.dynamodb.model.TableInfo;
import org.opensearch.dataprepper.plugins.source.dynamodb.stream.StreamCheckpointer;
import org.opensearch.dataprepper.plugins.source.dynamodb.utils.DynamoDBSourceAggregateMetrics;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.services.dynamodb.model.GetRecordsRequest;
import software.amazon.awssdk.services.dynamodb.model.GetRecordsResponse;
import software.amazon.awssdk.services.dynamodb.model.InternalServerErrorException;
import software.amazon.awssdk.services.dynamodb.streams.DynamoDbStreamsClient;

public class ShardConsumer
implements Runnable {
    private static final Logger LOG = LoggerFactory.getLogger(ShardConsumer.class);
    private static final Duration ACKNOWLEDGMENT_EXPIRY_INCREASE_TIME = Duration.ofMinutes(10L);
    private static final Duration ACKNOWLEDGMENT_PROGRESS_CHECK_INTERVAL = Duration.ofMinutes(3L);
    private static volatile boolean shouldStop = false;
    private static final Duration STREAM_EVENT_OVERLAP_TIME = Duration.ofMinutes(5L);
    private static final int MAX_GET_RECORD_ITEM_COUNT = 1000;
    private static final int GET_RECORD_INTERVAL_MILLS = 300;
    private static final int MINIMUM_GET_RECORD_INTERVAL_MILLS = 10;
    private static final long GET_RECORD_DELAY_THRESHOLD_MILLS = 15000L;
    private static final int DEFAULT_WAIT_FOR_EXPORT_INTERVAL_MILLS = 60000;
    private static final int DEFAULT_WAIT_COUNT_TO_CHECKPOINT = 3;
    private static final int DEFAULT_CHECKPOINT_INTERVAL_MILLS = 120000;
    static final Duration BUFFER_TIMEOUT = Duration.ofSeconds(60L);
    static final int DEFAULT_BUFFER_BATCH_SIZE = 1000;
    static final String SHARD_PROGRESS = "shardProgress";
    private final DynamoDbStreamsClient dynamoDbStreamsClient;
    private final StreamRecordConverter recordConverter;
    private final StreamCheckpointer checkpointer;
    private String shardIterator;
    private final String lastShardIterator;
    private final Instant startTime;
    private boolean waitForExport;
    private final AcknowledgementSet acknowledgementSet;
    private final Duration shardAcknowledgmentTimeout;
    private final String shardId;
    private final DynamoDBSourceAggregateMetrics dynamoDBSourceAggregateMetrics;
    private final Counter shardProgress;
    private long recordsWrittenToBuffer;

    private ShardConsumer(Builder builder) {
        this.shardProgress = builder.pluginMetrics.counter(SHARD_PROGRESS);
        this.dynamoDbStreamsClient = builder.dynamoDbStreamsClient;
        this.checkpointer = builder.checkpointer;
        this.shardIterator = builder.shardIterator;
        this.lastShardIterator = builder.lastShardIterator;
        this.startTime = builder.startTime == null ? Instant.MIN : builder.startTime.minus(STREAM_EVENT_OVERLAP_TIME);
        this.waitForExport = builder.waitForExport;
        BufferAccumulator bufferAccumulator = BufferAccumulator.create(builder.buffer, (int)1000, (Duration)BUFFER_TIMEOUT);
        this.recordConverter = new StreamRecordConverter((BufferAccumulator<Record<Event>>)bufferAccumulator, builder.tableInfo, builder.pluginMetrics, builder.streamConfig);
        this.acknowledgementSet = builder.acknowledgementSet;
        this.shardAcknowledgmentTimeout = builder.dataFileAcknowledgmentTimeout;
        this.shardId = builder.shardId;
        this.recordsWrittenToBuffer = 0L;
        this.dynamoDBSourceAggregateMetrics = builder.dynamoDBSourceAggregateMetrics;
    }

    public static Builder builder(DynamoDbStreamsClient dynamoDbStreamsClient, PluginMetrics pluginMetrics, DynamoDBSourceAggregateMetrics dynamoDBSourceAggregateMetrics, Buffer<Record<Event>> buffer, StreamConfig streamConfig) {
        return new Builder(dynamoDbStreamsClient, pluginMetrics, dynamoDBSourceAggregateMetrics, buffer, streamConfig);
    }

    @Override
    public void run() {
        LOG.debug("Shard Consumer start to run...");
        if (this.shouldSkip()) {
            this.shardProgress.increment();
            if (this.acknowledgementSet != null) {
                this.checkpointer.updateShardForAcknowledgmentWait(this.shardAcknowledgmentTimeout);
                this.acknowledgementSet.complete();
            }
            return;
        }
        if (this.acknowledgementSet != null) {
            this.addProgressCheck(this.acknowledgementSet);
        }
        long lastCheckpointTime = System.currentTimeMillis();
        String sequenceNumber = "";
        try {
            while (!shouldStop) {
                int interval;
                if (this.shardIterator == null) {
                    LOG.debug("Reached end of shard");
                    this.checkpointer.checkpoint(sequenceNumber);
                    break;
                }
                if (System.currentTimeMillis() - lastCheckpointTime > 120000L) {
                    LOG.debug("{} records written to buffer for shard {}", (Object)this.recordsWrittenToBuffer, (Object)this.shardId);
                    if (this.acknowledgementSet != null) {
                        this.checkpointer.updateShardForAcknowledgmentWait(this.shardAcknowledgmentTimeout);
                    } else {
                        this.checkpointer.checkpoint(sequenceNumber);
                    }
                    lastCheckpointTime = System.currentTimeMillis();
                }
                GetRecordsResponse response = this.callGetRecords(this.shardIterator);
                this.shardIterator = response.nextShardIterator();
                if (!response.records().isEmpty()) {
                    sequenceNumber = ((software.amazon.awssdk.services.dynamodb.model.Record)response.records().get(response.records().size() - 1)).dynamodb().sequenceNumber();
                    Instant lastEventTime = ((software.amazon.awssdk.services.dynamodb.model.Record)response.records().get(response.records().size() - 1)).dynamodb().approximateCreationDateTime();
                    if (lastEventTime.isBefore(this.startTime)) {
                        LOG.debug("Get {} events before start time, ignore...", (Object)response.records().size());
                        continue;
                    }
                    if (this.waitForExport) {
                        this.checkpointer.checkpoint(sequenceNumber);
                        this.waitForExport();
                        this.waitForExport = false;
                    }
                    List<software.amazon.awssdk.services.dynamodb.model.Record> records = response.records().stream().filter(record -> record.dynamodb().approximateCreationDateTime().isAfter(this.startTime)).collect(Collectors.toList());
                    this.recordConverter.writeToBuffer(this.acknowledgementSet, records);
                    this.shardProgress.increment();
                    this.recordsWrittenToBuffer += (long)records.size();
                    long delay = System.currentTimeMillis() - lastEventTime.toEpochMilli();
                    interval = delay > 15000L ? 10 : 300;
                } else {
                    interval = 300;
                    this.shardProgress.increment();
                }
                try {
                    Thread.sleep(interval);
                }
                catch (InterruptedException e) {
                    throw new RuntimeException(e);
                }
            }
            if (shouldStop) {
                LOG.warn("Processing for shard {} was interrupted by a shutdown signal, giving up shard", (Object)this.shardId);
                this.checkpointer.checkpoint(sequenceNumber);
                throw new RuntimeException("Consuming shard was interrupted from shutdown");
            }
            if (this.acknowledgementSet != null) {
                this.checkpointer.updateShardForAcknowledgmentWait(this.shardAcknowledgmentTimeout);
                this.acknowledgementSet.complete();
            }
            if (this.waitForExport) {
                this.waitForExport();
            }
        }
        catch (Exception exc) {
            if (this.acknowledgementSet != null) {
                this.acknowledgementSet.cancel();
            }
            throw exc;
        }
    }

    private GetRecordsResponse callGetRecords(String shardIterator) {
        GetRecordsRequest req = (GetRecordsRequest)GetRecordsRequest.builder().shardIterator(shardIterator).limit(Integer.valueOf(1000)).build();
        try {
            this.dynamoDBSourceAggregateMetrics.getStreamApiInvocations().increment();
            GetRecordsResponse response = this.dynamoDbStreamsClient.getRecords(req);
            return response;
        }
        catch (InternalServerErrorException ex) {
            this.dynamoDBSourceAggregateMetrics.getStream5xxErrors().increment();
            throw new RuntimeException(ex.getMessage());
        }
        catch (Exception e) {
            this.dynamoDBSourceAggregateMetrics.getStream4xxErrors().increment();
            throw new RuntimeException(e.getMessage());
        }
    }

    private void waitForExport() {
        LOG.debug("Start waiting for export to be done and loaded");
        int numberOfWaits = 0;
        while (!this.checkpointer.isExportDone()) {
            LOG.debug("Export is in progress, wait...");
            try {
                this.shardProgress.increment();
                Thread.sleep(60000L);
                if (++numberOfWaits % 3 != 0) continue;
                this.checkpointer.checkpoint(null);
            }
            catch (InterruptedException e) {
                LOG.error("Wait for export is interrupted ({})", (Object)e.getMessage());
                throw new RuntimeException("Wait for export is interrupted.");
            }
        }
    }

    private boolean shouldSkip() {
        if (this.lastShardIterator != null && !this.lastShardIterator.isEmpty()) {
            GetRecordsResponse response = this.callGetRecords(this.lastShardIterator);
            if (response.records().isEmpty()) {
                LOG.info("LastShardIterator is provided, but there is no Last Event Time, skip processing");
                return true;
            }
            Instant lastEventTime = ((software.amazon.awssdk.services.dynamodb.model.Record)response.records().get(response.records().size() - 1)).dynamodb().approximateCreationDateTime();
            if (lastEventTime.isBefore(this.startTime)) {
                LOG.info("LastShardIterator is provided, and Last Event Time is earlier than {}, skip processing", (Object)this.startTime);
                return true;
            }
            LOG.info("LastShardIterator is provided, and Last Event Time is later than {}, start processing", (Object)this.startTime);
            return false;
        }
        return false;
    }

    public static void stopAll() {
        shouldStop = true;
    }

    private void addProgressCheck(AcknowledgementSet acknowledgementSet) {
        acknowledgementSet.addProgressCheck(ignored -> acknowledgementSet.increaseExpiry(ACKNOWLEDGMENT_EXPIRY_INCREASE_TIME), ACKNOWLEDGMENT_PROGRESS_CHECK_INTERVAL);
    }

    static class Builder {
        private final DynamoDbStreamsClient dynamoDbStreamsClient;
        private final PluginMetrics pluginMetrics;
        private final DynamoDBSourceAggregateMetrics dynamoDBSourceAggregateMetrics;
        private final Buffer<Record<Event>> buffer;
        private TableInfo tableInfo;
        private StreamCheckpointer checkpointer;
        private String shardIterator;
        private String lastShardIterator;
        private Instant startTime;
        private boolean waitForExport;
        private String shardId;
        private AcknowledgementSet acknowledgementSet;
        private Duration dataFileAcknowledgmentTimeout;
        private StreamConfig streamConfig;

        public Builder(DynamoDbStreamsClient dynamoDbStreamsClient, PluginMetrics pluginMetrics, DynamoDBSourceAggregateMetrics dynamoDBSourceAggregateMetrics, Buffer<Record<Event>> buffer, StreamConfig streamConfig) {
            this.dynamoDbStreamsClient = dynamoDbStreamsClient;
            this.pluginMetrics = pluginMetrics;
            this.dynamoDBSourceAggregateMetrics = dynamoDBSourceAggregateMetrics;
            this.buffer = buffer;
            this.streamConfig = streamConfig;
        }

        public Builder tableInfo(TableInfo tableInfo) {
            this.tableInfo = tableInfo;
            return this;
        }

        public Builder shardId(String shardId) {
            this.shardId = shardId;
            return this;
        }

        public Builder checkpointer(StreamCheckpointer checkpointer) {
            this.checkpointer = checkpointer;
            return this;
        }

        public Builder shardIterator(String shardIterator) {
            this.shardIterator = shardIterator;
            return this;
        }

        public Builder lastShardIterator(String lastShardIterator) {
            this.lastShardIterator = lastShardIterator;
            return this;
        }

        public Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public Builder waitForExport(boolean waitForExport) {
            this.waitForExport = waitForExport;
            return this;
        }

        public Builder acknowledgmentSet(AcknowledgementSet acknowledgementSet) {
            this.acknowledgementSet = acknowledgementSet;
            return this;
        }

        public Builder acknowledgmentSetTimeout(Duration dataFileAcknowledgmentTimeout) {
            this.dataFileAcknowledgmentTimeout = dataFileAcknowledgmentTimeout;
            return this;
        }

        public ShardConsumer build() {
            return new ShardConsumer(this);
        }
    }
}

